# Student Group Assignment System

## Overview

This document describes the implementation of the student group assignment system in the CreActiHUB platform. This functionality allows instructors to create student groups based on various criteria (lab sections, days, time slots, etc.) and assign specific quizzes to these groups with custom due dates and penalty rates.

**Documentation date:** May 14, 2025

## Table of Contents

1. [Purpose](#purpose)
2. [Database Structure](#database-structure)
3. [System Components](#system-components)
4. [Workflow](#workflow)
5. [User Guide](#user-guide)
6. [Technical Details](#technical-details)
7. [Security Considerations](#security-considerations)
8. [Future Extensions](#future-extensions)

## Purpose

The student group assignment system was designed to solve the following problems:

- Allow instructors to organize students into logical groups (sections, days, times)
- Facilitate mass assignment of quizzes to entire groups rather than individually
- Provide specific due dates and penalty rates for different groups
- Improve class management for courses with multiple sections or lab groups

## Database Structure

### New Tables

#### `student_groups`

This table stores information about student groups.

| Field | Type | Description |
|-------|------|-------------|
| `group_id` | int(11) AUTO_INCREMENT | Unique group identifier |
| `group_name` | varchar(255) | Group name |
| `group_type` | enum('lab_section','day_slot','time_slot','custom') | Group type |
| `group_description` | text | Group description |
| `course_id` | int(11) | Course ID the group belongs to |
| `created_by` | int(11) | User ID of the group creator |
| `group_status` | tinyint(1) DEFAULT '1' | Group status (1=active, 0=inactive) |
| `created_date` | datetime DEFAULT CURRENT_TIMESTAMP | Creation date |

#### `student_group_members`

This table manages student membership in groups.

| Field | Type | Description |
|-------|------|-------------|
| `sgm_id` | int(11) AUTO_INCREMENT | Unique membership identifier |
| `sgm_group_id` | int(11) | Group ID |
| `sgm_user_id` | int(11) | Student ID |
| `sgm_status` | tinyint(1) DEFAULT '1' | Membership status |
| `sgm_added_by` | int(11) | User ID who added the student |
| `sgm_created_date` | datetime DEFAULT CURRENT_TIMESTAMP | Creation date |

### Related Existing Tables

#### `quiz_user_assignment`

This table already exists and is used to assign quizzes to users. Now it will also be used to assign quizzes to groups.

| Field | Type | Description |
|-------|------|-------------|
| `id` | int(11) AUTO_INCREMENT | Unique identifier |
| `quiz_id` | int(11) | Quiz ID |
| `user_id` | int(11) | User ID (0 if assignment to group) |
| `group_id` | int(11) | Group ID (if assignment to group) |
| `due_date` | datetime | Due date |
| `penalty_rate` | decimal(5,2) | Penalty rate (% per day) |

## System Components

### Models

1. **Model_student_group**: Handles creation, editing, and deletion of student groups.
2. **Model_student_group_member**: Manages adding and removing students from groups.
3. **Model_quiz_user_assignment** (existing, extended): Now includes functionality to assign quizzes to entire groups.

### Controllers

1. **Student_group**: Controller for managing student groups.
2. **Quiz** (existing, modified): Includes new functions to assign quizzes to groups.

### Views

1. **admin/student_group/list**: List of existing groups.
2. **admin/student_group/add**: Form to create new groups.
3. **admin/student_group/edit**: Form to edit existing groups.
4. **admin/student_group/manage_members**: Interface to add/remove students from a group.
5. **Modal** in quiz views: To assign quizzes to specific groups.

## Workflow

### Group Creation and Management

1. The instructor accesses the "Student Groups" section in the admin panel.
2. Creates a new group specifying name, type (lab section, day, etc.), course, and description.
3. In the group list, the instructor can view, edit, or delete existing groups.
4. To manage members, the instructor clicks the "Manage Members" button for a group.
5. On the member management screen, the instructor can add or remove students from the group.

### Assigning Quizzes to Groups

1. The instructor accesses the "Quizzes" section in the admin panel.
2. In the list of quizzes, the instructor clicks the "Assign to Group" button for a specific quiz.
3. In the modal that opens, selects the group, sets the due date, and the penalty rate.
4. On save, the quiz is assigned to all students in the selected group.

### Student Quiz View

1. The student logs in and accesses the "My Quizzes" section.
2. Sees all quizzes assigned directly or through the groups they belong to.
3. For each quiz, sees the specific due date and any other relevant information.
4. If attempting a quiz after the due date, the configured penalty will apply.

## User Guide

### For Instructors

#### Create a New Group

1. Navigate to Admin > Student Groups.
2. Click "Add New Group".
3. Fill out the form with the group name, type, course, and description.
4. Click "Save Group".

#### Add Students to a Group

1. In the group list, click the "Manage Members" button for the desired group.
2. In the form on the left, select a student from the drop-down list.
3. Click "Add to Group".
4. The student will appear in the member list on the right.

#### Assign a Quiz to a Group

1. Navigate to Admin > Quizzes.
2. In the quiz list, click the "Assign to Group" button for the desired quiz.
3. In the modal, select the group to assign the quiz to.
4. Set the due date and the penalty rate per day of delay.
5. Click "Assign Quiz".

### For Students

#### View Assigned Quizzes

1. Log in to your student account.
2. Navigate to "My Quizzes".
3. You will see all quizzes assigned directly to you or through groups you belong to.
4. Each quiz will show its due date.

#### Take a Quiz

1. In the quiz list, click on the quiz you wish to take.
2. If the current date is after the due date, you will be informed that a penalty may apply.
3. Complete the quiz as usual.

## Technical Details

### Roles and Permissions

The system uses existing user roles:

- **LECTURER_USER (1)**: Instructors who can create groups and assign quizzes.
- **CUSTOMER_USER (0)**: Students who are assigned to groups and take quizzes.
- **Administrators** (user_is_admin = 1): Have full access to all functionalities.

Permissions are controlled by checking the user type in each controller:

```php
if (!$this->user_is_admin && $this->user_type != LECTURER_USER) {
    redirect('admin/dashboard/access_denied');
}
```

### Penalty Calculation

The system uses the existing `calculate_penalty()` method in `Model_quiz_user_assignment` to determine the penalty for late submissions:

```php
public function calculate_penalty($days_late, $penalty_rate) {
    if ($days_late <= 0 || $penalty_rate <= 0) {
        return 0;
    }
    
    // Calculate penalty
    $penalty = $days_late * $penalty_rate / 100; // Convert percentage to decimal
    
    // Limit penalty to 100%
    return min($penalty, 1);
}
```

### Integration with Existing System

The system integrates with the existing quiz assignment framework, using the `group_id` field in the `quiz_user_assignment` table. When a quiz is assigned to a group, a record is created with:

- `user_id = 0` (indicates group assignment, not individual)
- `group_id = [Group ID]`

When a student accesses their quizzes, the system checks both direct assignments and group assignments for the groups they belong to.

## Security Considerations

1. **Permission Checks**: All critical operations (create/edit/delete groups, assign students, assign quizzes) verify that the user is an instructor or administrator.

2. **Data Validation**: All forms implement validation both client- and server-side.

3. **CSRF Protection**: CSRF tokens are used in all forms to prevent cross-site request forgery.

4. **Data Access**: Students can only see quizzes assigned directly to them or to their groups. The system implements checks to prevent unauthorized access.

5. **Ownership Verification**: Instructors can only manage groups they themselves created, unless they are administrators.

## Future Extensions

1. **Bulk Import**: Allow instructors to import students into groups from CSV files.

2. **Automatic Grouping**: Implement algorithms to automatically assign students to groups based on criteria such as capacity, schedules, etc.

3. **Notifications**: Send notifications to students when they are added to a group or assigned a new quiz.

4. **Group Statistics**: Provide comparative analytics of performance between different groups.

5. **Advanced Customization**: Allow configuration of different quiz parameters (time, number of attempts) per group.

---

## Technical Implementation

### Developer Instructions

1. Run the SQL scripts to create the `student_groups` and `student_group_members` tables.
2. Add the model files: `Model_student_group.php` and `Model_student_group_member.php`.
3. Implement the modifications in `Model_quiz_user_assignment.php`.
4. Add the `Student_group.php` controller and make changes to existing controllers.
5. Create the views for managing groups and members.
6. Modify existing views to include functionality for assigning quizzes to groups.

### Recommended Tests

1. Verify creation, editing, and deletion of groups.
2. Check adding and removing students from groups.
3. Test assigning quizzes to groups and setting specific due dates.
4. Verify that students only see the quizzes assigned to them.
5. Check the calculation of penalties for late submissions.

---

For any further questions about this implementation, contact the development team.
