/**
 * Question Bank Central
 * JavaScript to manage all functionalities of the centralized question bank
 */

$(document).ready(function() {
    // Global variables
    var selectedQuestions = [];
    var questionBankTable;
    
    // DataTable initialization
    initQuestionBankTable();
    
    // Set up event handlers
    setupEventHandlers();
    
    /**
     * Initializes the question bank DataTable
     */
    function initQuestionBankTable() {
        if (typeof questionBank === 'undefined' || !questionBank.urls || !questionBank.urls.getQuestions) {
            console.error('The global variable "questionBank" is not defined or is missing the getQuestions URL.');
            $('#question_bank_table').before('<div class="alert alert-danger">Error: The questionBank URLs configuration is not available. Contact the administrator.</div>');
            return;
        }
        questionBankTable = $('#datatable_ajax').DataTable({
            "processing": true,
            "serverSide": true,
            "ajax": {
                "url": questionBank.urls.getQuestions,
                "type": "POST",
                "data": function(d) {
                    d.course_id = $('#filter_course').val();
                    d.subject_id = $('#filter_subject').val();
                    d.question_type = $('#filter_type').val();
                    d.search_text = $('#search_questions').val();
                    d.show_only_unassigned = $('#show_only_unassigned').is(':checked') ? 1 : 0;
                }
            },
            "columns": [
                // ID con checkbox
                { 
                    "data": "qb_id",
                    "render": function(data, type, row) {
                        return '<input type="checkbox" class="question-checkbox" value="' + data + '"> ' + data;
                    },
                    "orderable": true,
                    "width": "5%"
                },
                // Question text
                { 
                    "data": "qb_text",
                    "orderable": true,
                    "width": "25%"
                },
                // Question type
                { 
                    "data": "qb_type",
                    "orderable": true,
                    "width": "10%"
                },
                // Course
                { 
                    "data": "qb_course_id",
                    "orderable": true,
                    "width": "10%"
                },
                // Subject
                { 
                    "data": "qb_subject_id",
                    "orderable": true,
                    "width": "10%"
                },
                // Is bonus
                { 
                    "data": "qb_is_bonus",
                    "orderable": true,
                    "width": "5%"
                },
                // Assigned to quizzes
                { 
                    "data": "assigned_quizzes",
                    "orderable": false,
                    "width": "15%"
                },
                // Estado
                { 
                    "data": "qb_status",
                    "orderable": true,
                    "width": "5%"
                },
                // Acciones
                { 
                    "data": "actions",
                    "orderable": false,
                    "width": "15%"
                }
            ],
            "order": [[0, "desc"]],
            "pageLength": 10,
            "language": {
                "search": "_INPUT_",
                "searchPlaceholder": "Search...",
                "emptyTable": "No questions available",
                "info": "Showing _START_ to _END_ of _TOTAL_ questions",
                "infoEmpty": "No questions available",
                "infoFiltered": "(filtered from _MAX_ total questions)",
                "lengthMenu": "Show _MENU_ questions",
                "zeroRecords": "No matching questions found"
            },
            "drawCallback": function() {
                // Restaurar checkboxes seleccionados
                $('.question-checkbox').each(function() {
                    var questionId = $(this).val();
                    if (selectedQuestions.indexOf(questionId) !== -1) {
                        $(this).prop('checked', true);
                    }
                });
                
                // Actualizar estados de botones
                updateButtonStates();
                
                // Configurar tooltips si están disponibles
                if (typeof $().tooltip === 'function') {
                    $('[data-toggle="tooltip"]').tooltip();
                }
            }
        });
    }
    
    /**
     * Configures event handlers for all interactive elements
     */
    function setupEventHandlers() {
        // Filter handling
        $('#filter_course, #filter_subject, #filter_type, #show_only_unassigned').change(function() {
            questionBankTable.ajax.reload();
        });
        
        // Text search with delay
        var searchTimeout;
        $('#search_questions').on('keyup', function() {
            clearTimeout(searchTimeout);
            searchTimeout = setTimeout(function() {
                questionBankTable.ajax.reload();
            }, 500);
        });
        
        // Load subjects when a course is selected (quiz_bank.js pattern)
        $('#filter_course').on('change', function() {
            var courseId = $(this).val();
            var subjectSelect = $('#filter_subject');
            subjectSelect.html('<option value="">Loading...</option>');
            if (courseId) {
                $.ajax({
                    url: questionBank.urls.getSubjects,
                    type: 'POST',
                    data: {course_id: courseId},
                    dataType: 'json',
                    success: function(response) {
                        subjectSelect.empty();
                        if (response.status && response.subjects) {
                            subjectSelect.append('<option value="">Select Subject</option>');
                            $.each(response.subjects, function(id, name) {
                                subjectSelect.append('<option value="' + id + '">' + name + '</option>');
                            });
                        } else {
                            subjectSelect.append('<option value="">No subjects available</option>');
                        }
                    },
                    error: function() {
                        subjectSelect.empty();
                        subjectSelect.append('<option value="">Error loading subjects</option>');
                    }
                });
            } else {
                subjectSelect.html('<option value="">Select Subject</option>');
            }
        });
        
        // Individual checkbox selection
        $(document).on('change', '.question-checkbox', function() {
            var questionId = $(this).val();
            
            if ($(this).is(':checked')) {
                // Add if not already selected
                if (selectedQuestions.indexOf(questionId) === -1) {
                    selectedQuestions.push(questionId);
                }
            } else {
                // Remove if selected
                var index = selectedQuestions.indexOf(questionId);
                if (index !== -1) {
                    selectedQuestions.splice(index, 1);
                }
            }
            
            updateButtonStates();
        });
        
        // Select all button
        $('#btn_select_all').click(function() {
            $('.question-checkbox').prop('checked', true).trigger('change');
        });
        
        // Clear selection button
        $('#btn_clear_selection').click(function() {
            $('.question-checkbox').prop('checked', false).trigger('change');
            selectedQuestions = [];
            updateButtonStates();
        });
        
        // Assign selected button
        $('#btn_assign_selected').click(function() {
            if (selectedQuestions.length > 0) {
                showAssignModal(selectedQuestions);
            }
        });
        
        // Assign individual button
        $(document).on('click', '.assign-question-btn', function() {
            var questionId = $(this).data('id');
            showAssignModal([questionId]);
        });
        
        // Delete selected button
        $('#btn_delete_selected').click(function() {
            if (selectedQuestions.length > 0) {
                if (confirm('Are you sure you want to delete ' + selectedQuestions.length + ' selected questions?')) {
                    // Implement delete logic for multiple questions
                    alert('Delete functionality to be implemented');
                }
            }
        });
        
        // Delete individual button
        $(document).on('click', '.delete-question-btn', function() {
            var questionId = $(this).data('id');
            
            if (confirm('Are you sure you want to delete this question?')) {
                $.ajax({
                    url: questionBank.urls.deleteQuestion + '/' + questionId,
                    type: 'POST',
                    dataType: 'json',
                    success: function(response) {
                        if (response.status) {
                            toastr.success(response.message);
                            questionBankTable.ajax.reload();
                            
                            // Remove from selection if it was selected
                            var index = selectedQuestions.indexOf(questionId.toString());
                            if (index !== -1) {
                                selectedQuestions.splice(index, 1);
                                updateButtonStates();
                            }
                        } else {
                            toastr.error(response.message);
                        }
                    },
                    error: function() {
                        toastr.error('Failed to delete question');
                    }
                });
            }
        });
        
        // Confirm question assignment
        $('#btnConfirmAssign').click(function() {
            var quizId = $('#quiz_id').val();
            
            if (!quizId) {
                toastr.error('Please select a quiz');
                return;
            }
            
            // Validate that there are selected questions
            if (!selectedQuestions.length) {
                toastr.error('Please select at least one question to assign.');
                return;
            }
            // Assign selected questions to quiz
            $.ajax({
                url: questionBank.urls.assignQuestions,
                type: 'POST',
                data: {
                    quiz_id: quizId,
                    // Send as JSON string for greater robustness
                    question_ids: JSON.stringify(selectedQuestions)
                },
                dataType: 'json',
                success: function(response) {
                    if (response.status) {
                        toastr.success(response.message);
                        $('#assignQuizModal').modal('hide');
                        questionBankTable.ajax.reload();
                    } else {
                        toastr.error(response.message);
                    }
                },
                error: function() {
                    toastr.error('Failed to assign questions');
                }
            });
        });
    }
    
    /**
     * Updates the states of bulk action buttons according to the selection
     */
    function updateButtonStates() {
        var hasSelected = selectedQuestions.length > 0;
        
        $('#btn_assign_selected').prop('disabled', !hasSelected);
        $('#btn_delete_selected').prop('disabled', !hasSelected);
        
        // Update selected questions counter in modal if it's visible
        if ($('#assignQuizModal').is(':visible')) {
            $('#selectedQuestionsCount').text(selectedQuestions.length);
        }
    }
    
    /**
     * Shows the modal to assign questions to a quiz
     * @param {Array} questionIds - Array of question IDs to assign
     */
    function showAssignModal(questionIds) {
        // Load available quizzes
        $.ajax({
            url: questionBank.urls.getQuizzes,
            type: 'POST',
            dataType: 'json',
            beforeSend: function() {
                $('#quiz_id').html('<option value="">Loading quizzes...</option>');
            },
            success: function(response) {
                if (response.status && response.quizzes.length > 0) {
                    var options = '<option value="">Select Quiz</option>';
                    
                    $.each(response.quizzes, function(i, quiz) {
                        var quizInfo = quiz.name;
                        if (quiz.course) {
                            quizInfo += ' (' + quiz.course + ')';
                        }
                        
                        options += '<option value="' + quiz.id + '">' + quizInfo + '</option>';
                    });
                    
                    $('#quiz_id').html(options);
                } else {
                    $('#quiz_id').html('<option value="">No quizzes available</option>');
                }
            },
            error: function() {
                $('#quiz_id').html('<option value="">Error loading quizzes</option>');
                toastr.error('Failed to load quizzes');
            }
        });
        
        // Update selected questions information
        $('#selectedQuestionsCount').text(questionIds.length);
        $('#selectedQuestionsInfo').show();
        
        // Show modal
        $('#assignQuizModal').modal('show');
    }
});
