/**
 * Question Bank Management
 * Handles all functionality related to quiz question bank, including:
 * - Loading questions via AJAX
 * - Filtering questions by various criteria
 * - Managing question selection and batch operations
 * - Assigning questions to quizzes
 */

$(document).ready(function() {
    // Initialize variables
    var selectedQuestions = [];
    var questionBankTable;
    
    // Initialize DataTable for question bank
    initQuestionBankTable();
    
    // Setup event handlers
    setupEventHandlers();
    
    /**
     * Initialize the question bank DataTable
     */
    function initQuestionBankTable() {
        // Fix: Hide loading overlay when loading is complete
        $('#question_bank_table').on('processing.dt', function(e, settings, processing) {
            if (!processing) {
                $('.dataTables_processing').hide();
                $('#preloader2').hide(); // Hide custom preloader
            }
        });
        questionBankTable = $('#question_bank_table').DataTable({
            // Standard DataTables layout, filter hidden by CSS
            "dom": "<'row'<'col-sm-12'tr>><'row'<'col-sm-5'l><'col-sm-4'i><'col-sm-3'p>>",
            "scrollX": true,

            "processing": true,
            "serverSide": true,
            "ajax": {
                "url": window.QuestionBankConfig.urls.getQuestionBankData,
                "type": "POST",
                "data": function(d) {
                    d.course_id = $('#filter_course').val();
                    d.subject_id = $('#filter_subject').val();
                    d.question_type = $('#filter_type').val();
                    d.search_text = $('#search_questions').val();
                    // For maximum compatibility with DataTables, also update the search.value field
                    d.search = d.search || {};
                    d.search.value = $('#search_questions').val();
                    // Send the filter for only unassigned questions
                    d.show_only_unassigned = $('#show_only_unassigned').is(':checked') ? 1 : 0;
                }
            },
            "columns": [
    { // 0: Checkbox
        "data": 0,
        "render": function(data, type, row) {
            return data; // The backend already returns the checkbox HTML
        }
    },
    { // 1: ID
        "data": 1
    },
    { // 2: Question
        "data": 2
    },
    { // 3: Assigned courses
        "data": 3
    },
    { // 4: Assigned quizzes
        "data": 4
    },
    { // 5: Subject
        "data": 5
    },
    { // 6: Type
        "data": 6
    },
    { // 7: Base Points
        "data": 7,
        "orderable": false
    },
    { // 8: Is bonus?
        "data": 8
    },
    { // 9: Status
        "data": 9
    },
    { // 10: Actions
        "data": 10,
        "orderable": false,
        "searchable": false
    }
],
            "order": [[0, "desc"]],
            "pageLength": 10,
            "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]]
        });
    }
    
    /**
     * Setup event handlers for the question bank
     */
    function setupEventHandlers() {
        // Course filter change
        $('#filter_course').change(function() {
            var courseId = $(this).val();
            loadSubjects(courseId);
            questionBankTable.ajax.reload();
        });
        
        // Subject filter change
        $('#filter_subject').change(function() {
            questionBankTable.ajax.reload();
        });
        
        // Type filter change
        $('#filter_type').change(function() {
            questionBankTable.ajax.reload();
        });
        
        // Search by text (on enter or on change)
        $('#search_questions').on('keypress', function(e) {
            if (e.which == 13) { // Enter key
                questionBankTable.ajax.reload();
            }
        }).on('change', function() {
            questionBankTable.ajax.reload();
        });
        
        // Select all questions
        $('#select_all_questions').change(function() {
            var checked = $(this).prop('checked');
            $('.question-checkbox').prop('checked', checked);
            updateSelectedQuestions();
        });
        
        // Individual question selection
        $(document).on('change', '.question-checkbox', function() {
            updateSelectedQuestions();
        });
        
        // Select all button (only visible questions on the current page)
        $('#btn_select_all').click(function() {
            $('#question_bank_table tbody .question-checkbox').prop('checked', true);
            updateSelectedQuestions();
        });
        // Clear selection button
        $('#btn_clear_selection').click(function() {
            $('#question_bank_table tbody .question-checkbox').prop('checked', false);
            updateSelectedQuestions();
        });
        // Assign selected questions to a quiz button
        $('#btn_assign_selected').click(function() {
            if (selectedQuestions.length > 0) {
                showAssignQuizModal();
            } else {
                toastr.warning('Select at least one question to assign.');
            }
        });
        // Delete selected questions
        $('#btn_delete_selected').click(function() {
            if (selectedQuestions.length > 0) {
                if (confirm('Are you sure you want to delete the selected questions? This action cannot be undone.')) {
                    deleteSelectedQuestions();
                }
            } else {
                toastr.warning('Select at least one question to delete.');
            }
        });
        
        // Individual question delete
        $(document).on('click', '.delete-question-btn', function() {
            var questionId = $(this).data('id');
            if (confirm('Are you sure you want to delete this question? This action cannot be undone.')) {
                deleteQuestion(questionId);
            }
        });
        
        // Individual question assignment (not implemented here, only multiple)
        
        // Button to load questions from a deleted quiz for cleaning. Only independent question management.
        
        // Drag & drop eliminated for cleaning. Only independent question management.
        
        // Show only unassigned change
        $('#show_only_unassigned').change(function() {
            questionBankTable.ajax.reload();
        });
    }
    
    /**
     * Load subjects based on selected course
     * @param {int} courseId - The course ID
     */
    function loadSubjects(courseId) {
        if (!courseId) {
            $('#filter_subject').prop('disabled', true).html('<option value="">Select Course First</option>');
            return;
        }
        $('#filter_subject').prop('disabled', false).html('<option value="">Loading...</option>');
        $.ajax({
            url: window.QuestionBankConfig.urls.getSubjectList,
            type: 'POST',
            data: { course_id: courseId },
            dataType: 'json',
            success: function(response) {
                var options = '<option value="">All Subjects</option>';
                if (response && response.length > 0) {
                    $.each(response, function(i, item) {
                        options += '<option value="' + item.ITEMID + '">' + item.ITEMDESC + '</option>';
                    });
                }
                $('#filter_subject').html(options).prop('disabled', false);
            },
            error: function(xhr, status, error) {
                $('#filter_subject').html('<option value="">All Subjects</option>').prop('disabled', false);
                console.error('Error loading subjects:', error);
            }
        });
    }
    
    /**
     * Update the selected questions array and button states
     */
    function updateSelectedQuestions() {
        selectedQuestions = [];
        $('.question-checkbox:checked').each(function() {
            selectedQuestions.push($(this).val());
        });
        
        // Update button states
        var hasSelected = selectedQuestions.length > 0;
        $('#btn_assign_selected').prop('disabled', !hasSelected);
        $('#btn_copy_selected').prop('disabled', !hasSelected);
        $('#btn_delete_selected').prop('disabled', !hasSelected);
    }
    
    /**
     * Show modal for assigning questions to a quiz
     */
    function showAssignQuizModal() {
        console.log('Opening assign quiz modal...');
        
        // First, load the quizzes directly - before creating the modal
        $.ajax({
            url: window.QuestionBankConfig.urls.getQuizzes,
            type: 'POST', 
            dataType: 'json',
            success: function(response) {
                console.log('Quiz data received:', response);
                
                // Generate select options
                var options = '<option value="">Select Quiz</option>';
                if (response && response.status && response.quizzes && response.quizzes.length > 0) {
                    console.log('Found ' + response.quizzes.length + ' quizzes to display');
                    $.each(response.quizzes, function(i, quiz) {
                        options += '<option value="' + quiz.id + '">' + quiz.name + '</option>';
                    });
                } else {
                    console.log('No quizzes found or invalid format:', response);
                    options = '<option value="">No quizzes available</option>';
                }
                
                // Remove previous modal if it exists
                if ($('#assignQuizModal').length > 0) {
                    $('#assignQuizModal').remove();
                }
                
                // Create modal with options already included
                var modalHtml = '<div class="modal fade" id="assignQuizModal" tabindex="-1" role="dialog" aria-labelledby="assignQuizModalLabel">'+
                    '<div class="modal-dialog" role="document">'+
                    '<div class="modal-content">'+
                    '<div class="modal-header">'+
                    '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>'+
                    '<h4 class="modal-title" id="assignQuizModalLabel">Assign Questions to Quiz</h4>'+
                    '</div>'+
                    '<div class="modal-body">'+
                    '<div class="form-group">'+
                    '<label for="quiz_id">Select Quiz:</label>'+
                    '<select id="quiz_id" class="form-control">'+
                    options+
                    '</select>'+
                    '</div>'+
                    '</div>'+
                    '<div class="modal-footer">'+
                    '<button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>'+
                    '<button type="button" class="btn btn-primary" id="btn_confirm_assign">Assign</button>'+
                    '</div>'+
                    '</div>'+
                    '</div>'+
                    '</div>';
                
                // Add modal to DOM
                $('body').append(modalHtml);
                
                // Show the modal
                $('#assignQuizModal').modal('show');
                
                // Handle click on assign button
                $('#btn_confirm_assign').off('click').on('click', function() {
                    var quizId = $('#quiz_id').val();
                    if (quizId) {
                        assignQuestionsToQuiz(selectedQuestions, quizId);
                        $('#assignQuizModal').modal('hide');
                    } else {
                        alert('Please select a quiz.');
                    }
                });
            },
            error: function(xhr, status, error) {
                console.error('Error loading quizzes:', error);
                
                // Create modal with error message
                var modalHtml = '<div class="modal fade" id="assignQuizModal" tabindex="-1" role="dialog" aria-labelledby="assignQuizModalLabel">'+
                    '<div class="modal-dialog" role="document">'+
                    '<div class="modal-content">'+
                    '<div class="modal-header">'+
                    '<button type="button" class="close" data-dismiss="modal" aria-label="Close"><span aria-hidden="true">&times;</span></button>'+
                    '<h4 class="modal-title" id="assignQuizModalLabel">Assign Questions to Quiz</h4>'+
                    '</div>'+
                    '<div class="modal-body">'+
                    '<div class="alert alert-danger">Error loading quizzes. Please try again later.</div>'+
                    '</div>'+
                    '<div class="modal-footer">'+
                    '<button type="button" class="btn btn-default" data-dismiss="modal">Close</button>'+
                    '</div>'+
                    '</div>'+
                    '</div>'+
                    '</div>';
                
                $('body').append(modalHtml);
                $('#assignQuizModal').modal('show');
            }
        });
    }
    
    /**
     * Load quizzes for the assignment modal
     */
    function loadQuizzesForModal() {
        console.log('Loading quizzes for modal...');
        
        // Try with POST first since the controller might be expecting POST
        $.ajax({
            url: window.QuestionBankConfig.urls.getQuizzes,
            type: 'POST',  // Change to POST to test
            dataType: 'json',
            beforeSend: function() {
                console.log('Sending POST request to get quizzes...');
            },
            success: function(response) {
                console.log('Raw response from get_quizzes:', response);
                processQuizResponse(response);
            },
            error: function(xhr, status, error) {
                console.error('POST Error loading quizzes:', error);
                console.log('Status code:', xhr.status);
                console.log('Response text:', xhr.responseText);
                
                // If it fails with POST, try with GET
                console.log('Trying with GET method as fallback...');
                
                $.ajax({
                    url: window.QuestionBankConfig.urls.getQuizzes,
                    type: 'GET',
                    dataType: 'json',
                    success: function(response) {
                        console.log('Raw response from GET get_quizzes:', response);
                        processQuizResponse(response);
                    },
                    error: function(xhr, status, error) {
                        console.error('GET Error loading quizzes:', error);
                        console.log('Status code:', xhr.status);
                        console.log('Response text:', xhr.responseText);
                        alert('Error loading quizzes. Please check the console for details.');
                        
                        // In case of error, load example data to test the interface
                        var demoOptions = '<option value="">Select Quiz</option>' +
                                         '<option value="1">Demo Quiz 1</option>' +
                                         '<option value="2">Demo Quiz 2</option>';
                        $('#quiz_id').html(demoOptions);
                    }
                });
            }
        });
    }
    
    function processQuizResponse(response) {
        var options = '<option value="">Select Quiz</option>';
        
        // Try different response formats
        if (response && response.status && response.quizzes && response.quizzes.length > 0) {
            console.log('Format 1: Found quizzes in response.quizzes');
            $.each(response.quizzes, function(i, quiz) {
                options += '<option value="' + quiz.id + '">' + quiz.name + '</option>';
            });
        } else if (response && Array.isArray(response) && response.length > 0) {
            console.log('Format 2: Response is direct array of quizzes');
            $.each(response, function(i, quiz) {
                // Try different properties that might contain the ID and name
                var id = quiz.id || quiz.quiz_id || quiz.q_id || i;
                var name = quiz.name || quiz.quiz_name || quiz.q_name || quiz.title || 'Quiz ' + (i+1);
                options += '<option value="' + id + '">' + name + '</option>';
            });
        } else if (response && response.quizzes && Array.isArray(response.quizzes)) {
            console.log('Format 3: Empty quizzes array or different structure');
            console.log('Quizzes:', response.quizzes);
        } else {
            console.log('Unrecognized response format:', response);
        }
        
        $('#quiz_id').html(options);
    }
    
    /**
     * Assign selected questions to a quiz
     * @param {Array} questionIds - Array of question IDs
     * @param {int} quizId - The quiz ID
     */
    function assignQuestionsToQuiz(questionIds, quizId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.assignQuestions,
            type: 'POST',
            data: JSON.stringify({
                questions: questionIds.map(function(id) {
                    return {
                        quiz_id: quizId,
                        question_id: id,
                        table_name: 'course_quiz'
                    };
                })
            }),
            contentType: 'application/json',
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    toastr.success('Questions successfully assigned to quiz!');
                    questionBankTable.ajax.reload();
                    // Clear selection
                    $('#select_all_questions').prop('checked', false);
                    updateSelectedQuestions();
                } else {
                    var errorMsg = 'Error assigning questions: ' + (response.message || 'Unknown error');
                // If there are detailed errors, show them in a modal or alert
                // Unify general message and details in a single toastr
                if (response.errors && Array.isArray(response.errors) && response.errors.length > 0) {
                    var htmlDetails = '<ul>';
                    response.errors.forEach(function(err) {
                        htmlDetails += '<li><b>ID:</b> ' + err.question_id + ' - ' + err.message + '</li>';
                    });
                    htmlDetails += '</ul>';
                    toastr.error(errorMsg + htmlDetails);
                } else {
                    toastr.error(errorMsg);
                }
                }
            },
            error: function(xhr, status, error) {
                toastr.error('Error assigning questions: ' + error);
            }
        });
    }
    
    /**
     * Delete a single question
     * @param {int} questionId - The question ID to delete
     */
    function deleteQuestion(questionId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.deleteQuestion,
            type: 'POST',
            data: { question_id: questionId },
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    toastr.success('Question deleted successfully!');
                    questionBankTable.ajax.reload();
                } else {
                    toastr.error('Error deleting question: ' + (response.message || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                toastr.error('Error deleting question: ' + error);
            }
        });
    }
    
    /**
     * Delete multiple selected questions
     */
    function deleteSelectedQuestions() {
    var totalQuestions = selectedQuestions.length;
    var deletedCount = 0;
    var errorCount = 0;
    
    selectedQuestions.forEach(function(questionId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.deleteQuestion,
            type: 'POST',
            data: { question_id: questionId },
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    deletedCount++;
                } else {
                    errorCount++;
                }
                // Check if all requests are complete
                if (deletedCount + errorCount === totalQuestions) {
                    if (errorCount === 0) {
                        toastr.success('All selected questions deleted successfully!');
                    } else {
                        toastr.warning('Deleted ' + deletedCount + ' questions. Failed to delete ' + errorCount + ' questions.');
                    }
                    questionBankTable.ajax.reload();
                    $('#select_all_questions').prop('checked', false);
                    updateSelectedQuestions();
                }
            },
            error: function(xhr, status, error) {
                errorCount++;
                // Check if all requests are complete
                if (deletedCount + errorCount === totalQuestions) {
                    toastr.warning('Deleted ' + deletedCount + ' questions. Failed to delete ' + errorCount + ' questions.');
                    questionBankTable.ajax.reload();
                    $('#select_all_questions').prop('checked', false);
                    updateSelectedQuestions();
                }
            }
        });
    });
}

    
    /**
     * Load questions for a specific quiz
     * @param {int} quizId - The quiz ID
     */
    function loadQuizQuestions(quizId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.getQuizQuestions,
            type: 'GET',
            data: { quiz_id: quizId },
            dataType: 'json',
            success: function(response) {
                renderQuizQuestions(quizId, response);
            },
            error: function(xhr, status, error) {
                $('#quiz_questions_container').html(
                    '<div class="alert alert-danger">' +
                    '<strong>Error:</strong> Failed to load quiz questions. ' + error +
                    '</div>'
                );
            }
        });
    }
    
    /**
     * Render quiz questions in the assigned questions tab
     * @param {int} quizId - The quiz ID
     * @param {Array} questions - The questions data
     */
    function renderQuizQuestions(quizId, questions) {
        if (!questions || questions.length === 0) {
            $('#quiz_questions_container').html(
                '<div class="alert alert-warning">' +
                '<strong>No questions:</strong> This quiz has no questions assigned to it.' +
                '</div>'
            );
            return;
        }
        
        var html = '<div class="quiz-questions-list">';
        html += '<h4>Quiz Questions</h4>';
        html += '<p>You can reorder questions by dragging them.</p>';
        html += '<ul class="list-group quiz-questions-sortable" data-quiz-id="' + quizId + '">';
        
        questions.forEach(function(question, index) {
            html += '<li class="list-group-item" data-question-id="' + question.cq_id + '">';
            html += '<div class="row">';
            
            // Question number and drag handle
            html += '<div class="col-md-1">';
            html += '<span class="badge">' + (index + 1) + '</span> ';
            html += '<i class="fa fa-arrows-alt drag-handle"></i>';
            html += '</div>';
            
            // Question content
            html += '<div class="col-md-9">';
            html += '<div class="question-text">' + question.cq_question + '</div>';
            html += '<div class="question-meta">';
            html += '<small>Type: ' + getQuestionTypeName(question.cq_type) + ' | ';
            html += 'Points: ' + question.cq_gain_point;
            if (question.cq_is_bonus == 1) {
                html += ' + ' + (question.cq_bonus_points || 0) + ' bonus';
            }
            html += '</small>';
            html += '</div>';
            html += '</div>';
            
            // Actions
            html += '<div class="col-md-2 text-right">';
            html += '<a href="' + window.QuestionBankConfig.urls.editQuestion + question.cq_id + '" class="btn btn-xs btn-info"><i class="fa fa-edit"></i></a> ';
            html += '<a href="javascript:void(0);" class="btn btn-xs btn-danger remove-from-quiz" data-quiz-id="' + quizId + '" data-question-id="' + (question.cq_question_bank_id || question.qb_id) + '"><i class="fa fa-times"></i></a>';
            html += '</div>';
            
            html += '</div>';
            html += '</li>';
        });
        
        html += '</ul>';
        html += '</div>';
        
        $('#quiz_questions_container').html(html);
        
        // Initialize sortable
        $('.quiz-questions-sortable').sortable({
            handle: '.drag-handle',
            update: function(event, ui) {
                saveQuestionOrder(quizId);
            }
        });
        
        // Attach remove handlers
        $('.remove-from-quiz').click(function() {
            var questionId = $(this).data('question-id');
            var quizId = $(this).data('quiz-id');
            
            if (confirm('Are you sure you want to remove this question from the quiz?')) {
                removeQuestionFromQuiz(questionId, quizId);
            }
        });
    }
    
    /**
     * Get question type name from ID
     * @param {string} typeId - The question type ID
     * @return {string} The type name
     */
    function getQuestionTypeName(typeId) {
        var types = {
            '1': 'Multiple Choice',
            '2': 'True/False',
            '3': 'Essay',
            '4': 'Short Answer'
        };
        return types[typeId] || 'Unknown';
    }
    
    /**
     * Remove a question from a quiz
     * @param {int} questionId - The question ID
     * @param {int} quizId - The quiz ID
     */
    function removeQuestionFromQuiz(questionId, quizId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.removeQuestionFromQuiz,
            type: 'POST',
            data: { question_id: questionId, quiz_id: quizId },
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    toastr.success('Question removed from quiz successfully!');
                    loadQuizQuestions(quizId);
                } else {
                    toastr.error('Error removing question: ' + (response.message || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                toastr.error('Error removing question: ' + error);
            }
        });
    }
    
    /**
     * Save the order of questions in a quiz
     * @param {int} quizId - The quiz ID
     */
    function saveQuestionOrder(quizId) {
        var questionIds = [];
        $('.quiz-questions-sortable li').each(function() {
            questionIds.push($(this).data('question-id'));
        });
        
        $.ajax({
            url: window.QuestionBankConfig.urls.saveQuestionOrder,
            type: 'POST',
            data: { quiz_id: quizId, question_ids: questionIds },
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    toastr.success('Question order saved successfully!');
                } else {
                    toastr.error('Error saving question order: ' + (response.message || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                toastr.error('Error saving question order: ' + error);
            }
        });
    }
    
    /**
     * Setup drag and drop functionality for quiz questions
     */
    function setupDragAndDrop() {
        // Make question rows draggable
        $(document).on('mouseenter', '#question_bank_table tbody tr', function() {
            if (!$(this).data('draggable-initialized')) {
                $(this).draggable({
                    helper: 'clone',
                    appendTo: 'body',
                    zIndex: 999,
                    cursor: 'move',
                    start: function(event, ui) {
                        $(ui.helper).css('width', $(this).width());
                        $(ui.helper).addClass('dragging-question');
                    }
                });
                $(this).data('draggable-initialized', true);
            }
        });
        
        // Make quiz selector droppable
        $('#filter_quiz').droppable({
            accept: '#question_bank_table tbody tr',
            hoverClass: 'droppable-hover',
            drop: function(event, ui) {
                var quizId = $(this).val();
                if (!quizId) {
                    toastr.warning('Please select a quiz first.');
                    return;
                }
                
                var questionId = ui.draggable.find('input.question-checkbox').val();
                if (questionId) {
                    assignQuestionsDragDrop([questionId], quizId);
                }
            }
        });
    }
    
    /**
     * Assign questions via drag and drop
     * @param {Array} questionIds - Array of question IDs
     * @param {int} quizId - The quiz ID
     */
    function assignQuestionsDragDrop(questionIds, quizId) {
        $.ajax({
            url: window.QuestionBankConfig.urls.assignQuestions,
            type: 'POST',
            data: JSON.stringify({
                questions: questionIds.map(function(id) {
                    return {
                        quiz_id: quizId,
                        question_id: id,
                        table_name: 'course_quiz'
                    };
                })
            }),
            contentType: 'application/json',
            dataType: 'json',
            success: function(response) {
                if (response.status) {
                    toastr.success('Question successfully assigned to quiz!');
                    // If the assigned tab is active and the quiz is selected, reload the questions
                    if ($('#tab_assigned').hasClass('active') && $('#filter_quiz').val() == quizId) {
                        loadQuizQuestions(quizId);
                    }
                } else {
                    toastr.error('Error assigning question: ' + (response.message || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                toastr.error('Error assigning question: ' + error);
            }
        });
    }
});
