<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Model_quiz_questions extends MY_Model {
    protected $_table = 'tn_quiz_questions';
    protected $_primary_key = 'qq_id';
    protected $_primary_filter = 'intval';
    protected $_order_by = 'qq_order';
    
    function __construct() {
        parent::__construct();
    }
    
    /**
     * Asigna una pregunta a un quiz
     * @param int $quiz_id ID del quiz
     * @param int $question_id ID de la pregunta
     * @param array $options Opciones adicionales (orden, puntos, etc.)
     * @return int|bool ID de la relaciu00f3n o false en caso de error
     */
    public function assign_question($quiz_id, $question_id, $options = array()) {
        log_message('error', '[DEBUG] assign_question called with quiz_id=' . print_r($quiz_id, true) . ' question_id=' . print_r($question_id, true) . ' options=' . print_r($options, true));
        // Verificar si la pregunta ya está asignada al quiz
        log_message('error', '[DEBUG] Verificando si pregunta ya está asignada: quiz_id=' . $quiz_id . ', question_id=' . $question_id);
        $existing = $this->db->where('qq_quiz_id', $quiz_id)
                           ->where('qq_question_id', $question_id)
                           ->get($this->_table)
                           ->row();
        
        if ($existing) {
            // La pregunta ya estu00e1 asignada a este quiz, actualizar opciones si es necesario
            if (!empty($options)) {
                $this->db->where('qq_id', $existing->qq_id);
                $this->db->update($this->_table, $options);
            }
            return $existing->qq_id;
        }
        
        // Determinar el orden si no se especifica
        if (!isset($options['qq_order']) || $options['qq_order'] === null) {
            $max_order = $this->db->select_max('qq_order')
                              ->where('qq_quiz_id', $quiz_id)
                              ->get($this->_table)
                              ->row();
            $options['qq_order'] = $max_order ? $max_order->qq_order + 1 : 1;
        }
        
        // Obtener puntos del banco de preguntas para usar como valor inicial
        $this->db->select('qb_gain_point');
        $this->db->from('tn_question_bank');
        $this->db->where('qb_id', $question_id);
        $question = $this->db->get()->row();
        
        $bank_points = ($question && $question->qb_gain_point > 0) ? floatval($question->qb_gain_point) : 1.0;
        
        $data = array(
            'qq_quiz_id' => $quiz_id,
            'qq_question_id' => $question_id,
            'qq_points' => isset($options['qq_points']) ? $options['qq_points'] : $bank_points,
            'qq_created_on' => date('Y-m-d H:i:s')
        );
        
        // Agregar opciones adicionales
        foreach ($options as $key => $value) {
            $data[$key] = $value;
        }
        
        // Insertar relación
        log_message('error', '[DEBUG] Insertando en tn_quiz_questions: ' . print_r($data, true));
        $this->db->insert($this->_table, $data);
        return $this->db->insert_id();
    }
    
    /**
     * Asigna mu00faltiples preguntas a un quiz
     * @param int $quiz_id ID del quiz
     * @param array $question_ids Array de IDs de preguntas
     * @param array $options Opciones adicionales
     * @return array Resultados de asignaciu00f3n
     */
    public function assign_multiple_questions($quiz_id, $question_ids, $options = array()) {
        $results = array(
            'success' => 0,
            'error' => 0,
            'duplicates' => 0,
            'ids' => array()
        );
        
        foreach ($question_ids as $question_id) {
            $existing = $this->db->where('qq_quiz_id', $quiz_id)
                               ->where('qq_question_id', $question_id)
                               ->get($this->_table)
                               ->row();
            
            if ($existing) {
                $results['duplicates']++;
                continue;
            }
            
            $result = $this->assign_question($quiz_id, $question_id, $options);
            if ($result) {
                $results['success']++;
                $results['ids'][] = $result;
            } else {
                $results['error']++;
            }
        }
        
        return $results;
    }
    
    /**
     * Obtiene todas las preguntas asignadas a un quiz con la informaciu00f3n completa
     * @param int $quiz_id ID del quiz
     * @return array Preguntas asignadas con datos completos
     */
    public function get_quiz_questions($quiz_id) {
        $this->db->select('qq.*, qb.qb_id, qb.qb_text, qb.qb_type, qb.qb_gain_point, qq_order as question_order, qq_is_bonus_override');
        $this->db->from($this->_table . ' qq');
        $this->db->join('tn_question_bank qb', 'qb.qb_id = qq.qq_question_id', 'left');
        $this->db->where('qq.qq_quiz_id', $quiz_id);
        $this->db->order_by('qq.qq_order', 'ASC');
        $questions = $this->db->get()->result_array();

        // Load question bank model if not already loaded
        $CI =& get_instance();
        $CI->load->model('Model_question_bank');

        foreach ($questions as &$question) {
            $question['type'] = $CI->Model_question_bank->getQuestionType(isset($question['qb_type']) ? $question['qb_type'] : 0);
        }
        unset($question);
        return $questions;
    }
    
    /**
     * Obtiene todas las preguntas asignadas a un quiz con la informaciu00f3n completa
     * @param int $quiz_id ID del quiz
     * @return array Preguntas asignadas con datos completos
     */
    public function get_quiz_questions_complete($quiz_id) {
        // 1. Primero obtener preguntas manuales asignadas (fijas)
        $manual_questions = $this->get_quiz_questions($quiz_id);
        
        // 2. Luego obtener preguntas dinámicas de categorías
        $dynamic_questions = $this->get_category_questions_for_quiz($quiz_id);
        
        // 3. Combinar ambas listas
        return array_merge($manual_questions, $dynamic_questions);
    }
    
    /**
     * Obtiene preguntas dinámicas de categorías para un quiz
     * @param int $quiz_id ID del quiz
     * @return array Preguntas dinámicas con jerarquía de puntos
     */
    public function get_category_questions_for_quiz($quiz_id) {
        $this->load->model('model_quiz_category_config');
        
        // Obtener configuraciones de categorías para este quiz
        $category_configs = $this->model_quiz_category_config->get_category_configs($quiz_id);
        
        $dynamic_questions = array();
        
        foreach ($category_configs as $config) {
            // Obtener preguntas de esta categoría
            $this->db->select('qb.qb_id, qb.qb_text, qb.qb_type, qb.qb_gain_point, qb.qb_is_bonus, qb.qb_category_id');
            $this->db->from('tn_question_bank qb');
            $this->db->where('qb.qb_category_id', $config['qcc_category_id']);
            $this->db->where('qb.qb_status', 1);
            
            // Aplicar selección aleatoria si está configurada
            if ($config['qcc_random_selection']) {
                $this->db->order_by('RAND()');
            }
            
            $this->db->limit($config['qcc_num_questions']);
            $questions = $this->db->get()->result_array();
            
            // Procesar cada pregunta con jerarquía de puntos para categorías
            foreach ($questions as $question) {
                // Jerarquía para preguntas dinámicas: qcc_points_per_question → qb_gain_point → 1.0
                if (isset($config['qcc_points_per_question']) && $config['qcc_points_per_question'] > 0) {
                    $effective_points = floatval($config['qcc_points_per_question']);
                } elseif (isset($question['qb_gain_point']) && $question['qb_gain_point'] > 0) {
                    $effective_points = floatval($question['qb_gain_point']);
                } else {
                    $effective_points = 1.0;
                }
                
                $dynamic_questions[] = array_merge($question, array(
                    'qq_points' => null, // No tiene asignación manual
                    'effective_points' => $effective_points,
                    'source' => 'category', // Indicar que viene de categoría
                    'category_config' => $config
                ));
            }
        }
        
        return $dynamic_questions;
    }
    
    /**
     * Elimina la asignaciu00f3n de una pregunta a un quiz
     * @param int $quiz_id ID del quiz
     * @param int $question_id ID de la pregunta
     * @return bool u00c9xito de la operaciu00f3n
     */
    public function unassign_question($quiz_id, $question_id) {
        return $this->db->where('qq_quiz_id', $quiz_id)
                      ->where('qq_question_id', $question_id)
                      ->delete($this->_table);
    }
    
    /**
     * Reordena las preguntas en un quiz
     * @param int $quiz_id ID del quiz
     * @param array $order_data Array con [question_id => orden]
     * @return bool u00c9xito de la operaciu00f3n
     */
    public function reorder_questions($quiz_id, $order_data) {
        $success = true;
        
        foreach ($order_data as $question_id => $order) {
            $result = $this->db->where('qq_quiz_id', $quiz_id)
                             ->where('qq_question_id', $question_id)
                             ->update($this->_table, array('qq_order' => $order));
            if (!$result) {
                $success = false;
            }
        }
        
        return $success;
    }
    
    /**
     * Get the assignment record for a specific question in a quiz
     * @param int $quiz_id ID del quiz
     * @param int $question_id ID de la pregunta
     * @return array|null Assignment record or null if not found
     */
    public function get_question_assignment($quiz_id, $question_id) {
        $this->db->select('*');
        $this->db->from($this->_table);
        $this->db->where('qq_quiz_id', $quiz_id);
        $this->db->where('qq_question_id', $question_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row_array();
        }
        
        return null;
    }
    
    /**
     * Update points for a specific question assignment
     * @param int $quiz_id ID del quiz
     * @param int $question_id ID de la pregunta
     * @param float $points New points value
     * @return bool Success of the operation
     */
    public function update_question_points($quiz_id, $question_id, $points) {
        return $this->db->where('qq_quiz_id', $quiz_id)
                       ->where('qq_question_id', $question_id)
                       ->update($this->_table, array(
                           'qq_points' => floatval($points),
                           'qq_modified_on' => date('Y-m-d H:i:s')
                       ));
    }
}
