<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Model_quiz_category_questions extends MY_Model {
    protected $_table = 'quiz_category_questions';
    protected $_field_prefix = 'qcq_';
    protected $_pk = 'qcq_id';
    public $pagination_params = array();
    public $relations = array();
    public $dt_params = array();
    public $_per_page = 20;

    function __construct() {
        parent::__construct();
    }

    /**
     * Assign a question to a category
     * @param int $category_id Category ID
     * @param int $question_id Question ID
     * @return int|bool ID of the assignment or false on failure
     */
    public function assign_question($category_id, $question_id) {
        // Check if the assignment already exists
        $existing = $this->db->where('qcq_category_id', $category_id)
                           ->where('qcq_question_id', $question_id)
                           ->get($this->_table)
                           ->row();
        
        if ($existing) {
            // Already assigned
            return $existing->qcq_id;
        }
        
        // Create new assignment
        $data = array(
            'qcq_category_id' => $category_id,
            'qcq_question_id' => $question_id,
            'qcq_created_on' => date('Y-m-d H:i:s')
        );
        
        $this->db->insert($this->_table, $data);
        return $this->db->insert_id();
    }

    /**
     * Assign multiple questions to a category
     * @param int $category_id Category ID
     * @param array $question_ids Array of question IDs
     * @return array Results of assignments
     */
    public function assign_multiple_questions($category_id, $question_ids) {
        $results = array(
            'success' => 0,
            'error' => 0,
            'duplicates' => 0,
            'ids' => array()
        );
        
        foreach ($question_ids as $question_id) {
            $existing = $this->db->where('qcq_category_id', $category_id)
                               ->where('qcq_question_id', $question_id)
                               ->get($this->_table)
                               ->row();
            
            if ($existing) {
                $results['duplicates']++;
                continue;
            }
            
            $result = $this->assign_question($category_id, $question_id);
            if ($result) {
                $results['success']++;
                $results['ids'][] = $result;
            } else {
                $results['error']++;
            }
        }
        
        return $results;
    }

    /**
     * Remove a question from a category
     * @param int $category_id Category ID
     * @param int $question_id Question ID
     * @return bool Success of operation
     */
    public function remove_question($category_id, $question_id) {
        return $this->db->where('qcq_category_id', $category_id)
                      ->where('qcq_question_id', $question_id)
                      ->delete($this->_table);
    }

    /**
     * Get all questions in a category
     * @param int $category_id Category ID
     * @return array Questions in the category
     */
    public function get_category_questions($category_id) {
        $this->db->select('qcq.*, qb.*');
        $this->db->from($this->_table . ' qcq');
        $this->db->join('question_bank qb', 'qb.qb_id = qcq.qcq_question_id', 'left');
        $this->db->where('qcq.qcq_category_id', $category_id);
        $this->db->order_by('qb.qb_id', 'ASC');
        $questions = $this->db->get()->result_array();

        // Load question bank model for type mapping
        $CI =& get_instance();
        $CI->load->model('Model_question_bank');

        // Map question types to human-readable format
        foreach ($questions as &$question) {
            $question['type'] = $CI->Model_question_bank->getQuestionType(
                isset($question['qb_type']) ? $question['qb_type'] : 0
            );
        }
        unset($question);
        
        return $questions;
    }

    /**
     * Get random questions from a category
     * @param int $category_id Category ID
     * @param int $limit Number of questions to retrieve
     * @return array Random questions from the category
     */
    public function get_random_questions($category_id, $limit = 1) {
        $this->db->select('qcq.*, qb.*');
        $this->db->from($this->_table . ' qcq');
        $this->db->join('question_bank qb', 'qb.qb_id = qcq.qcq_question_id', 'left');
        $this->db->where('qcq.qcq_category_id', $category_id);
        $this->db->order_by('RAND()'); // Random order
        $this->db->limit($limit);
        $questions = $this->db->get()->result_array();

        // Load question bank model for type mapping
        $CI =& get_instance();
        $CI->load->model('Model_question_bank');

        // Map question types
        foreach ($questions as &$question) {
            $question['type'] = $CI->Model_question_bank->getQuestionType(
                isset($question['qb_type']) ? $question['qb_type'] : 0
            );
        }
        unset($question);
        
        return $questions;
    }

    /**
     * Check if a question is in a category
     * @param int $category_id Category ID
     * @param int $question_id Question ID
     * @return bool Whether question is in category
     */
    public function is_in_category($category_id, $question_id) {
        $this->db->where('qcq_category_id', $category_id);
        $this->db->where('qcq_question_id', $question_id);
        $count = $this->db->count_all_results($this->_table);
        return $count > 0;
    }

    /**
     * Get all categories for a question
     * @param int $question_id Question ID
     * @return array Categories that contain the question
     */
    public function get_question_categories($question_id) {
        $this->db->select('qc.*');
        $this->db->from($this->_table . ' qcq');
        $this->db->join('quiz_categories qc', 'qc.qc_id = qcq.qcq_category_id', 'left');
        $this->db->where('qcq.qcq_question_id', $question_id);
        $this->db->where('qc.qc_status', STATUS_ACTIVE);
        $this->db->order_by('qc.qc_name', 'ASC');
        
        return $this->db->get()->result_array();
    }
}
