<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Model_quiz_category_config extends MY_Model {
    protected $_table = 'quiz_category_config';
    protected $_field_prefix = 'qcc_';
    protected $_pk = 'qcc_id';
    public $pagination_params = array();
    public $relations = array();
    public $dt_params = array();
    public $_per_page = 20;

    function __construct() {
        parent::__construct();
    }

    /**
     * Configure a category for a quiz
     * @param int $quiz_id Quiz ID
     * @param int $category_id Category ID
     * @param int $num_questions Number of questions to include
     * @param bool $random_selection Whether to select questions randomly
     * @return int|bool ID of the configuration or false on failure
     */
    public function configure_category($quiz_id, $category_id, $num_questions = 1, $random_selection = true, $points_per_question = null) {
        // Check if configuration already exists
        $existing = $this->db->where('qcc_quiz_id', $quiz_id)
                           ->where('qcc_category_id', $category_id)
                           ->get($this->_table)
                           ->row();
        
        $now = date('Y-m-d H:i:s');
        
        if ($existing) {
            // Update existing configuration
            $data = array(
                'qcc_num_questions' => $num_questions,
                'qcc_random_selection' => $random_selection ? 1 : 0,
                'qcc_modified_on' => $now
            );
            
            // Solo actualizar puntos si se proporciona un valor
            if ($points_per_question !== null) {
                $data['qcc_points_per_question'] = floatval($points_per_question);
            }
            
            $this->db->where('qcc_id', $existing->qcc_id);
            $this->db->update($this->_table, $data);
            return $existing->qcc_id;
        }
        
        // Create new configuration
        $data = array(
            'qcc_quiz_id' => $quiz_id,
            'qcc_category_id' => $category_id,
            'qcc_num_questions' => $num_questions,
            'qcc_random_selection' => $random_selection ? 1 : 0,
            'qcc_created_on' => $now,
            'qcc_modified_on' => $now
        );
        
        // Agregar puntos si se proporciona
        if ($points_per_question !== null) {
            $data['qcc_points_per_question'] = floatval($points_per_question);
        }
        
        $this->db->insert($this->_table, $data);
        return $this->db->insert_id();
    }

    /**
     * Get all category configurations for a quiz
     * @param int $quiz_id Quiz ID
     * @return array Category configurations for the quiz
     */
    public function get_quiz_configurations($quiz_id) {
        $this->db->select('qcc.*, qc.qc_name, qc.qc_description');
        $this->db->from($this->_table . ' qcc');
        $this->db->join('quiz_categories qc', 'qc.qc_id = qcc.qcc_category_id', 'left');
        $this->db->where('qcc.qcc_quiz_id', $quiz_id);
        $this->db->where('qc.qc_status', STATUS_ACTIVE);
        $this->db->order_by('qcc.qcc_id', 'ASC');
        
        return $this->db->get()->result_array();
    }

    /**
     * Remove a category configuration from a quiz
     * @param int $quiz_id Quiz ID
     * @param int $category_id Category ID
     * @return bool Success of operation
     */
    public function remove_configuration($quiz_id, $category_id) {
        return $this->db->where('qcc_quiz_id', $quiz_id)
                      ->where('qcc_category_id', $category_id)
                      ->delete($this->_table);
    }

    /**
     * Update quiz category configurations in batch
     * @param int $quiz_id Quiz ID
     * @param array $configurations Array of category configurations
     * @return array Results of operation
     */
    public function update_configurations($quiz_id, $configurations) {
        $results = array(
            'success' => 0,
            'error' => 0
        );
        
        // First, delete all existing configurations
        $this->db->where('qcc_quiz_id', $quiz_id);
        $this->db->delete($this->_table);
        
        // Then, add the new configurations
        foreach ($configurations as $config) {
            if (empty($config['category_id'])) continue;
            
            $category_id = $config['category_id'];
            $num_questions = isset($config['num_questions']) && intval($config['num_questions']) > 0 ? 
                intval($config['num_questions']) : 1;
            $random_selection = isset($config['random_selection']) ? 
                (bool)$config['random_selection'] : true;
            
            $points_per_question = isset($config['points_per_question']) ? $config['points_per_question'] : null;
            $result = $this->configure_category($quiz_id, $category_id, $num_questions, $random_selection, $points_per_question);
            
            if ($result) {
                $results['success']++;
            } else {
                $results['error']++;
            }
        }
        
        return $results;
    }

    /**
     * Generate questions for a quiz based on its category configurations
     * @param int $quiz_id Quiz ID
     * @return array Selected questions for the quiz
     */
    public function generate_quiz_questions($quiz_id) {
        $CI =& get_instance();
        $CI->load->model('Model_quiz_category_questions');
        
        $questions = array();
        $configurations = $this->get_quiz_configurations($quiz_id);
        
        foreach ($configurations as $config) {
            $category_id = $config['qcc_category_id'];
            $num_questions = $config['qcc_num_questions'];
            $random_selection = $config['qcc_random_selection'];
            
            if ($random_selection) {
                // Get random questions from this category
                $category_questions = $CI->Model_quiz_category_questions->get_random_questions($category_id, $num_questions);
            } else {
                // Get sequential questions (first N questions)
                $category_questions = $CI->Model_quiz_category_questions->get_category_questions($category_id);
                $category_questions = array_slice($category_questions, 0, $num_questions);
            }
            
            // Add category questions to the final set
            foreach ($category_questions as $question) {
                // Add category info to each question for reference
                $question['category_id'] = $category_id;
                $question['category_name'] = $config['qc_name'];
                $questions[] = $question;
            }
        }
        
        return $questions;
    }

    /**
     * Count total questions that will be generated for a quiz
     * @param int $quiz_id Quiz ID
     * @return int Total number of questions
     */
    public function count_total_questions($quiz_id) {
        $this->db->select('SUM(qcc_num_questions) as total');
        $this->db->from($this->_table);
        $this->db->where('qcc_quiz_id', $quiz_id);
        $result = $this->db->get()->row();
        
        return $result ? intval($result->total) : 0;
    }
}
