<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Model_quiz_categories extends MY_Model {
    protected $_table = 'quiz_categories';
    protected $_field_prefix = 'qc_';
    protected $_pk = 'qc_id';
    protected $_status_field = 'qc_status';
    public $pagination_params = array();
    public $relations = array();
    public $dt_params = array();
    public $_per_page = 20;

    function __construct() {
        // Call the Model constructor
        $this->pagination_params['fields'] = "qc_id,qc_name,qc_course_id,qc_subject_id,qc_status";
        parent::__construct();
    }

    /**
     * Get categories with optional filters
     * @param array $params Filter parameters
     * @param int $id Category ID (optional)
     * @return array Categories
     */
    public function get_data($params = array(), $id = 0) {
        // Join with course and subject tables for additional info
        $params['joins'][] = $this->join_course();
        $params['joins'][] = $this->join_subject();

        if (!isset($params['where'][$this->_status_field])) {
            $params['where_string'] = $this->_status_field . ' < 2';
        }

        if ($id) {
            return $this->find_by_pk($id, false, $params);
        } else {
            return $this->find_all($params);
        }
    }

    /**
     * Get active categories
     * @param array $params Filter parameters
     * @param int $id Category ID (optional)
     * @return array Active categories
     */
    public function get_data_active($params = array(), $id = 0) {
        $params['where'][$this->_status_field] = STATUS_ACTIVE;
        return $this->get_data($params, $id);
    }

    /**
     * Join with course table
     * @param string $type Join type
     * @param string $append_joint Additional join conditions
     * @param string $prepend_joint Prepend join conditions
     * @return array Join configuration
     */
    public function join_course($type = "", $append_joint = "", $prepend_joint = "") {
        $joint = $prepend_joint . "qc_course_id = course_id " . $append_joint;
        return $this->prep_join("course", $joint, $type);
    }

    /**
     * Join with subject table
     * @param string $type Join type
     * @param string $append_joint Additional join conditions
     * @param string $prepend_joint Prepend join conditions
     * @return array Join configuration
     */
    public function join_subject($type = "", $append_joint = "", $prepend_joint = "") {
        $joint = $prepend_joint . "qc_subject_id = course_subject_id " . $append_joint;
        return $this->prep_join("course_subject", $joint, $type);
    }

    /**
     * Get categories for a specific course
     * @param int $course_id Course ID
     * @return array Categories
     */
    public function get_by_course($course_id) {
        $params = array();
        $params['where']['qc_course_id'] = $course_id;
        $params['where']['qc_status'] = STATUS_ACTIVE;
        $params['order'] = 'qc_name ASC';
        return $this->find_all($params);
    }

    /**
     * Get categories for a specific subject
     * @param int $subject_id Subject ID
     * @return array Categories
     */
    public function get_by_subject($subject_id) {
        $params = array();
        $params['where']['qc_subject_id'] = $subject_id;
        $params['where']['qc_status'] = STATUS_ACTIVE;
        $params['order'] = 'qc_name ASC';
        return $this->find_all($params);
    }

    /**
     * Count questions in a category
     * @param int $category_id Category ID
     * @return int Number of questions
     */
    public function count_questions($category_id) {
        $this->db->where('qcq_category_id', $category_id);
        return $this->db->count_all_results('quiz_category_questions');
    }

    /**
     * Field definitions for form generation
     * @param string $specific_field Specific field to return
     * @return array Field definitions
     */
    public function get_fields($specific_field = "") {
        $fields = array(
            'qc_id' => array(
                'table' => $this->_table,
                'name' => 'qc_id',
                'label' => 'ID #',
                'type' => 'hidden',
                'type_dt' => 'text',
                'attributes' => array(),
                'dt_attributes' => array("width" => "5%"),
                'js_rules' => '',
                'rules' => 'trim'
            ),

            'qc_name' => array(
                'table' => $this->_table,
                'name' => 'qc_name',
                'label' => 'Name',
                'type' => 'text',
                'attributes' => array(),
                'js_rules' => 'required',
                'rules' => 'required|trim|htmlentities'
            ),

            'qc_description' => array(
                'table' => $this->_table,
                'name' => 'qc_description',
                'label' => 'Description',
                'type' => 'textarea',
                'attributes' => array(),
                'js_rules' => '',
                'rules' => 'trim|htmlentities'
            ),

            'qc_course_id' => array(
                'table' => $this->_table,
                'name' => 'qc_course_id',
                'label' => 'Course',
                'type' => 'dropdown',
                'attributes' => array(
                    "class" => "ajax-populate",
                    "additional" => 'data-target="qc_subject_id"'
                ),
                'js_rules' => 'required',
                'rules' => 'required|trim|htmlentities'
            ),

            'qc_subject_id' => array(
                'table' => $this->_table,
                'name' => 'qc_subject_id',
                'label' => 'Subject',
                'type' => 'dropdown',
                'type_dt' => 'dropdown',
                'type_filter_dt' => 'dropdown',
                'attributes' => array(
                    "class" => "ajax-populate",
                    "additional" => ' 
                        data-populate-uri="get_subjects"
                        data-uri="quiz_categories"
                        data-dd_key="course_subject_id"
                        data-dd_value="course_subject_name"'
                ),
                'js_rules' => 'required',
                'rules' => 'required|trim'
            ),

            'qc_status' => array(
                'table' => $this->_table,
                'name' => 'qc_status',
                'label' => 'Status',
                'type' => 'switch',
                'type_dt' => 'switch',
                'type_filter_dt' => 'dropdown',
                'list_data' => array(),
                'default' => '1',
                'attributes' => array(),
                'dt_attributes' => array("width" => "7%"),
                'rules' => 'trim'
            ),
        );

        if ($specific_field) {
            return $fields[$specific_field];
        } else {
            return $fields;
        }
    }
}
