<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Model_question_bank extends MY_Model {
    protected $_table = 'tn_question_bank';
    protected $_primary_key = 'qb_id';
    protected $_primary_filter = 'intval';
    protected $_order_by = 'qb_id';
    
    function __construct() {
        parent::__construct();
    }
    
    /**
     * Obtener preguntas con filtros opcionales
     * @param array $params Parámetros de filtrado
     * @param array $pagination Configuración de paginación
     * @return array Preguntas que coinciden con los criterios
     */
    public function get_questions($params = array(), $pagination = null) {
        // Aplicar filtros si existen
        if (isset($params['course_id']) && !empty($params['course_id'])) {
            $this->db->where('qb_course_id', $params['course_id']);
        }
        
        if (isset($params['subject_id']) && !empty($params['subject_id'])) {
            $this->db->where('qb_subject_id', $params['subject_id']);
        }
        
        if (isset($params['type']) && !empty($params['type'])) {
            $this->db->where('qb_type', $params['type']);
        }
        
        if (isset($params['status'])) {
            $this->db->where('qb_status', $params['status']);
        }
        
        // Búsqueda por texto
        if (isset($params['search']) && !empty($params['search'])) {
            $this->db->like('qb_text', $params['search']);
        }
        
        // Búsqueda por tags
        if (isset($params['tags']) && !empty($params['tags'])) {
            $tags = explode(',', $params['tags']);
            foreach ($tags as $tag) {
                $this->db->like('qb_tags', trim($tag));
            }
        }
        
        // Aplicar paginación si se especifica
        if ($pagination) {
            if (isset($pagination['start']) && isset($pagination['length'])) {
                $this->db->limit($pagination['length'], $pagination['start']);
            }
        }
        
        // Ordenar por fecha más reciente por defecto
        $this->db->order_by('qb_created_on', 'DESC');
        
        $questions = $this->db->get($this->_table)->result_array();
        // Mapear el campo type para cada pregunta
        foreach ($questions as &$question) {
            $question['type'] = $this->getQuestionType(isset($question['qb_type']) ? $question['qb_type'] : 0);
        }
        unset($question);
        return $questions;
    }
    
    /**
     * Contar el total de preguntas que coinciden con los filtros
     * @param array $params Parámetros de filtrado
     * @return int Total de preguntas
     */
    public function count_questions($params = array()) {
        // Aplicar los mismos filtros que en get_questions
        if (isset($params['course_id']) && !empty($params['course_id'])) {
            $this->db->where('qb_course_id', $params['course_id']);
        }
        
        if (isset($params['subject_id']) && !empty($params['subject_id'])) {
            $this->db->where('qb_subject_id', $params['subject_id']);
        }
        
        if (isset($params['type']) && !empty($params['type'])) {
            $this->db->where('qb_type', $params['type']);
        }
        
        if (isset($params['status'])) {
            $this->db->where('qb_status', $params['status']);
        }
        
        // Búsqueda por texto
        if (isset($params['search']) && !empty($params['search'])) {
            $this->db->like('qb_text', $params['search']);
        }
        
        // Búsqueda por tags
        if (isset($params['tags']) && !empty($params['tags'])) {
            $tags = explode(',', $params['tags']);
            foreach ($tags as $tag) {
                $this->db->like('qb_tags', trim($tag));
            }
        }
        
        return $this->db->count_all_results($this->_table);
    }
    
    /**
     * Obtiene información sobre en qué quizzes está utilizada una pregunta
     * @param int $question_id ID de la pregunta
     * @return array Quizzes donde se usa esta pregunta
     */
    public function get_question_usage($question_id) {
        log_message('error', '[DEBUG get_question_usage] ID recibido: ' . print_r($question_id, true));
        // Construir la consulta para logging
        $this->db->select('qq.qq_id, qq.qq_quiz_id, q.quiz_name, c.course_title');
        $this->db->from('tn_quiz_questions qq');
        $this->db->join('quiz q', 'q.quiz_id = qq.qq_quiz_id', 'left');
        $this->db->join('course c', 'c.course_id = q.quiz_course_id', 'left');
        $this->db->where('qq.qq_question_id', $question_id);
        $sql = $this->db->get_compiled_select();
        log_message('error', '[DEBUG get_question_usage] SQL generado: ' . $sql);
        // RECONSTRUIR la consulta para get()
        $this->db->select('qq.qq_id, qq.qq_quiz_id, q.quiz_name, c.course_title');
        $this->db->from('tn_quiz_questions qq');
        $this->db->join('quiz q', 'q.quiz_id = qq.qq_quiz_id', 'left');
        $this->db->join('course c', 'c.course_id = q.quiz_course_id', 'left');
        $this->db->where('qq.qq_question_id', $question_id);
        $result = $this->db->get()->result_array();
    log_message('error', '[DEBUG get_question_usage] Resultado: ' . print_r($result, true));
        return $result;
    }
    
    /**
     * Guarda una pregunta nueva o actualiza una existente
     * @param array $data Datos de la pregunta
     * @param int $id ID de la pregunta (null para crear nueva)
     * @return int|bool ID de la pregunta o false en caso de error
     */
    public function save_question($data, $id = null) {
        // Establecer timestamp de modificación
        $data['qb_modified_on'] = date('Y-m-d H:i:s');
        
        if ($id) {
            // Actualizar pregunta existente
            $this->db->where('qb_id', $id);
            return $this->db->update($this->_table, $data) ? $id : false;
        } else {
            // Crear nueva pregunta
            $data['qb_created_on'] = date('Y-m-d H:i:s');
            $this->db->insert($this->_table, $data);
            return $this->db->insert_id();
        }
    }
    
    /**
     * Elimina una pregunta del banco de preguntas y todas sus asignaciones en quizzes
     * @param int $question_id ID de la pregunta
     * @return bool Éxito de la operación
     */
    public function delete_question($question_id) {
        // Eliminar asignaciones en quizzes
        $this->db->where('qq_question_id', $question_id);
        $this->db->delete('quiz_questions');
        // Eliminar la pregunta del banco
        $this->db->where('qb_id', $question_id);
        return $this->db->delete($this->_table);
    }

    /**
     * Map question type integer to human-readable string
     * @param int $type
     * @return string
     */
    public function getQuestionType($type)
    {
        $types = array(
            1 => 'Multiple Choice',
            2 => 'True/False',
            3 => 'Essay',
            4 => 'Short Answer',
            5 => 'Fill in the Blank',
        );
        $result = isset($types[$type]) ? $types[$type] : 'Unknown';
        log_message('error', '[DEBUG getQuestionType] Input: ' . print_r($type, true) . ' | Output: ' . $result);
        return $result;
    }
}
