<?php
if (!defined('BASEPATH'))
	exit('No direct script access allowed');

/**
 * Library for calculating quiz penalties
 *
 * @package		Quiz Penalty Calculator
 * @author		
 * @version		1.0
 * @since		Version 1.0 2025
 */

class Quiz_penalty_calculator {

    protected $CI;

    /**
     * Constructor
     */
    public function __construct() {
        // Get CI instance
        $this->CI =& get_instance();
        
        // Load required models
        $this->CI->load->model('model_quiz');
        $this->CI->load->model('model_quiz_user_assignment');
    }

    /**
     * Calculates the days late for a submission
     * 
     * @param int $quiz_id Quiz ID
     * @param int $user_id User ID
     * @param string $submission_date Submission date (YYYY-MM-DD)
     * @return int Number of days late (0 if on time)
     */
    public function calculate_days_late($quiz_id, $user_id, $submission_date) {
        // Look for custom assignment first
        $custom_assignment = $this->get_custom_assignment($quiz_id, $user_id);
        
        // If there is a custom assignment, use that due date
        if ($custom_assignment) {
            $due_date = $custom_assignment->due_date;
        } else {
            // If there is no custom assignment, use the general quiz due date
            $quiz = $this->CI->model_quiz->find_by_pk($quiz_id);
            if (!$quiz) {
                return 0; // If quiz not found, no late days
            }
            $due_date = $quiz->quiz_enddate;
        }
        
        // Convert dates to DateTime objects
        $due_datetime = new DateTime($due_date);
        $submission_datetime = new DateTime($submission_date);
        
        // If submission is before or on the due date, no late days
        if ($submission_datetime <= $due_datetime) {
            return 0;
        }
        
        // Calculate difference in days
        $interval = $submission_datetime->diff($due_datetime);
        return $interval->days;
    }

    /**
     * Calculates the penalty to apply
     * 
     * @param int $quiz_id Quiz ID
     * @param int $user_id User ID
     * @param int $days_late Days late
     * @return float Penalty percentage
     */
    public function calculate_penalty_percentage($quiz_id, $user_id, $days_late) {
        if ($days_late <= 0) {
            return 0; // No late days, no penalty
        }
        
        // Look for custom assignment first
        $custom_assignment = $this->get_custom_assignment($quiz_id, $user_id);
        
        // If there is a custom assignment, use that penalty rate
        if ($custom_assignment && $custom_assignment->penalty_rate !== null) {
            $daily_penalty = floatval($custom_assignment->penalty_rate);
        } else {
            // If there is no custom assignment, use the general quiz penalty rate
            $quiz = $this->CI->model_quiz->find_by_pk($quiz_id);
            if (!$quiz || !isset($quiz->quiz_daily_penalty)) {
                return 0; // If quiz not found or has no penalty, no penalty applied
            }
            $daily_penalty = floatval($quiz->quiz_daily_penalty);
        }
        
        // Calculate total penalty (percentage)
        $total_penalty = $daily_penalty * $days_late;
        
        // Limit penalty to 100%
        return min($total_penalty, 100);
    }

    /**
     * Calculates the final score after applying penalty
     * 
     * @param float $original_score Original score
     * @param float $penalty_percentage Penalty percentage
     * @return float Final score
     */
    public function calculate_final_score($original_score, $penalty_percentage) {
        if ($penalty_percentage <= 0) {
            return $original_score;
        }
        
        // Apply penalty
        $penalty_amount = ($original_score * $penalty_percentage) / 100;
        $final_score = $original_score - $penalty_amount;
        
        // Ensure score is not negative
        return max(0, $final_score);
    }

    /**
     * Gets the custom assignment for a user and quiz
     * 
     * @param int $quiz_id Quiz ID
     * @param int $user_id User ID
     * @return object|null Assignment object or null if it doesn't exist
     */
    protected function get_custom_assignment($quiz_id, $user_id) {
        $params = array(
            'where' => array(
                'quiz_id' => $quiz_id,
                'user_id' => $user_id,
                'status' => 1 // Only active assignments
            )
        );
        
        $assignments = $this->CI->model_quiz_user_assignment->find_all($params);
        if (!$assignments || count($assignments) == 0) {
            // Search by group if there is no individual assignment
            // First we need to get the user's groups
            $this->CI->load->model('model_user_group');
            $user_groups = $this->CI->model_user_group->get_user_groups($user_id);
            
            if ($user_groups) {
                foreach ($user_groups as $group) {
                    $params = array(
                        'where' => array(
                            'quiz_id' => $quiz_id,
                            'group_id' => $group->group_id,
                            'status' => 1
                        )
                    );
                    
                    $group_assignments = $this->CI->model_quiz_user_assignment->find_all($params);
                    if ($group_assignments && count($group_assignments) > 0) {
                        return $group_assignments[0]; // Use the first group assignment found
                    }
                }
            }
            
            return null;
        }
        
        return $assignments[0]; // Return the first assignment found
    }
}

/* End of file Quiz_penalty_calculator.php */
/* Location: ./application/libraries/Quiz_penalty_calculator.php */
