<?php
class Assignment_group_management extends MY_Controller {
    function __construct() {
        parent::__construct();
        $this->load->model('model_assignment_group');
        $this->load->model('model_assignment_group_member');
        $this->load->model('model_user');
        $this->load->model('model_course');
        
        // Solo administradores y profesores pueden acceder a esta sección
        if (!$this->user_is_admin && $this->user_type != LECTURER_USER) {
            redirect('admin/dashboard/access_denied');
        }
    }
    
    /**
     * Lista todos los grupos de estudiantes
     */
    public function index() {
        // Parámetros para recuperar grupos
        $params = array();
        
        // Filtrar solo grupos para estudiantes (no regulares)
        $params['where']['assignment_group_type !='] = 'regular';
        
        // Si es profesor, solo mostrar sus grupos
        if ($this->user_type == LECTURER_USER && !$this->user_is_admin) {
            // Obtener los cursos del profesor
            $courses = $this->db->select('course_id')
                              ->from('course')
                              ->where('course_instructor_id', $this->user_id)
                              ->get()
                              ->result_array();
            
            $course_ids = array_column($courses, 'course_id');
            
            if (!empty($course_ids)) {
                $params['where_in']['assignment_group_course_id'] = $course_ids;
            } else {
                // Si el profesor no tiene cursos, no debería ver grupos
                $params['where']['assignment_group_id'] = 0; // No dará resultados
            }
        }
        
        $data['groups'] = $this->model_assignment_group->find_all($params);
        $data['title'] = "Student Groups";
        $data['layout_title'] = "Manage Student Groups";
        
        $this->load_view("admin/assignment_group/list", $data);
    }
    
    /**
     * Formulario para añadir un nuevo grupo
     */
    public function add() {
        // Procesar formulario de creación de grupo
        if ($this->input->post('submit')) {
            // Obtener datos del formulario
            $form_data = $this->input->post('form_data');
            
            // Asegurarse de que el grupo no es de tipo regular
            if ($form_data['assignment_group_type'] == 'regular') {
                $form_data['assignment_group_type'] = 'custom';
            }
            
            // Si es profesor, verificar que el curso le pertenece
            if ($this->user_type == LECTURER_USER && !$this->user_is_admin) {
                $course = $this->model_course->find_by_pk($form_data['assignment_group_course_id']);
                if (!$course || $course->course_instructor_id != $this->user_id) {
                    $this->session->set_flashdata('error', 'You do not have permission to create groups for this course');
                    redirect('admin/assignment_group_management');
                }
            }
            
            // Validar y guardar
            if ($this->model_assignment_group->save($form_data)) {
                $this->session->set_flashdata('success', 'Group has been created successfully');
                redirect('admin/assignment_group_management');
            } else {
                $this->session->set_flashdata('error', 'Error creating group');
            }
        }
        
        // Obtener lista de cursos para dropdown
        if ($this->user_is_admin) {
            $courses = $this->model_course->find_all_list_active(array(), 'course_name');
        } else {
            // Si es profesor, solo mostrar sus cursos
            $params = array();
            $params['where']['course_instructor_id'] = $this->user_id;
            $courses = $this->model_course->find_all_list_active($params, 'course_name');
        }
        
        // Lista de tipos de grupo
        $group_types = array(
            'lab_section' => 'Laboratory Section',
            'day_slot' => 'Day Slot',
            'time_slot' => 'Time Slot',
            'custom' => 'Custom Group'
        );
        
        $data['courses'] = $courses;
        $data['group_types'] = $group_types;
        $data['title'] = "Add New Student Group";
        $data['layout_title'] = "Add New Student Group";
        
        $this->load_view("admin/assignment_group/add", $data);
    }
    
    /**
     * Formulario para editar un grupo existente
     */
    public function edit($group_id) {
        // Verificar si el usuario puede editar este grupo
        if (!$this->user_is_admin && !$this->model_assignment_group->can_manage_group($group_id, $this->user_id)) {
            $this->session->set_flashdata('error', 'You do not have permission to edit this group');
            redirect('admin/assignment_group_management');
        }
        
        // Obtener grupo
        $group = $this->model_assignment_group->find_by_pk($group_id);
        if (!$group) {
            $this->session->set_flashdata('error', 'Group not found');
            redirect('admin/assignment_group_management');
        }
        
        // Procesar formulario de edición
        if ($this->input->post('submit')) {
            $form_data = $this->input->post('form_data');
            
            // Asegurarse de que el grupo no es de tipo regular
            if ($form_data['assignment_group_type'] == 'regular') {
                $form_data['assignment_group_type'] = 'custom';
            }
            
            if ($this->model_assignment_group->update_by_pk($group_id, $form_data)) {
                $this->session->set_flashdata('success', 'Group has been updated successfully');
                redirect('admin/assignment_group_management');
            } else {
                $this->session->set_flashdata('error', 'Error updating group');
            }
        }
        
        // Obtener listas para dropdowns
        if ($this->user_is_admin) {
            $courses = $this->model_course->find_all_list_active(array(), 'course_name');
        } else {
            // Si es profesor, solo mostrar sus cursos
            $params = array();
            $params['where']['course_instructor_id'] = $this->user_id;
            $courses = $this->model_course->find_all_list_active($params, 'course_name');
        }
        
        $group_types = array(
            'lab_section' => 'Laboratory Section',
            'day_slot' => 'Day Slot',
            'time_slot' => 'Time Slot',
            'custom' => 'Custom Group'
        );
        
        $data['group'] = $group;
        $data['courses'] = $courses;
        $data['group_types'] = $group_types;
        $data['title'] = "Edit Student Group";
        $data['layout_title'] = "Edit Student Group: " . $group->assignment_group_name;
        
        $this->load_view("admin/assignment_group/edit", $data);
    }
    
    /**
     * Eliminar un grupo
     */
    public function delete($group_id) {
        // Verificar permisos
        if (!$this->user_is_admin && !$this->model_assignment_group->can_manage_group($group_id, $this->user_id)) {
            $this->session->set_flashdata('error', 'You do not have permission to delete this group');
            redirect('admin/assignment_group_management');
        }
        
        // Verificar que no es un grupo regular
        $group = $this->model_assignment_group->find_by_pk($group_id);
        if ($group && $group->assignment_group_type == 'regular') {
            $this->session->set_flashdata('error', 'Regular groups cannot be deleted from this section');
            redirect('admin/assignment_group_management');
        }
        
        // Eliminar grupo
        if ($group && $this->model_assignment_group->delete_by_pk($group_id)) {
            // También eliminar membresías de grupo
            $this->db->where('agm_group_id', $group_id);
            $this->db->delete('assignment_group_members');
            
            $this->session->set_flashdata('success', 'Group has been deleted successfully');
        } else {
            $this->session->set_flashdata('error', 'Error deleting group');
        }
        
        redirect('admin/assignment_group_management');
    }
    
    /**
     * Gestionar miembros de un grupo
     */
    public function manage_members($group_id) {
        // Verificar permisos
        if (!$this->user_is_admin && !$this->model_assignment_group->can_manage_group($group_id, $this->user_id)) {
            $this->session->set_flashdata('error', 'You do not have permission to manage this group');
            redirect('admin/assignment_group_management');
        }
        
        // Obtener grupo
        $group = $this->model_assignment_group->find_by_pk($group_id);
        if (!$group || $group->assignment_group_type == 'regular') {
            $this->session->set_flashdata('error', 'Invalid group or regular group');
            redirect('admin/assignment_group_management');
        }
        
        // Manejar adición de miembros
        if ($this->input->post('add_member')) {
            $user_id = $this->input->post('user_id');
            
            // Verificar si ya es miembro
            if (!$this->model_assignment_group_member->is_member($group_id, $user_id)) {
                $member_data = array(
                    'agm_group_id' => $group_id,
                    'agm_user_id' => $user_id,
                    'agm_added_by' => $this->user_id,
                    'agm_status' => 1
                );
                
                if ($this->model_assignment_group_member->save($member_data)) {
                    $this->session->set_flashdata('success', 'Student has been added to group');
                } else {
                    $this->session->set_flashdata('error', 'Error adding student to group');
                }
            } else {
                $this->session->set_flashdata('error', 'Student is already a member of this group');
            }
            
            redirect('admin/assignment_group_management/manage_members/' . $group_id);
        }
        
        // Manejar eliminación de miembros
        if ($this->input->post('remove_member')) {
            $member_id = $this->input->post('member_id');
            
            if ($this->model_assignment_group_member->delete_by_pk($member_id)) {
                $this->session->set_flashdata('success', 'Student has been removed from group');
            } else {
                $this->session->set_flashdata('error', 'Error removing student from group');
            }
            
            redirect('admin/assignment_group_management/manage_members/' . $group_id);
        }
        
        // Obtener miembros actuales del grupo
        $members = $this->model_assignment_group_member->get_group_members($group_id);
        
        // Obtener estudiantes para dropdown (solo de tipo CUSTOMER_USER)
        $params = array();
        $params['where']['user_type'] = CUSTOMER_USER;
        $params['where']['user_status'] = 1;
        $students = $this->model_user->find_all($params);
        
        $data['group'] = $group;
        $data['members'] = $members;
        $data['students'] = $students;
        $data['title'] = "Manage Group Members";
        $data['layout_title'] = "Manage Members: " . $group->assignment_group_name;
        
        $this->load_view("admin/assignment_group/manage_members", $data);
    }
    
    /**
     * Añadir varios estudiantes al mismo tiempo (AJAX)
     */
    public function bulk_add_members() {
        // Verificar si es una petición AJAX
        if (!$this->input->is_ajax_request()) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'Invalid request';
            echo json_encode($this->json_param);
            exit;
        }
        
        $group_id = $this->input->post('group_id');
        
        // Verificar permisos
        if (!$this->user_is_admin && !$this->model_assignment_group->can_manage_group($group_id, $this->user_id)) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'Permission denied';
            echo json_encode($this->json_param);
            exit;
        }
        
        // Obtener datos
        $student_ids = $this->input->post('student_ids');
        
        if (empty($student_ids)) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'No students selected';
            echo json_encode($this->json_param);
            exit;
        }
        
        // Procesar cada estudiante
        $added_count = 0;
        foreach ($student_ids as $user_id) {
            // Verificar si ya es miembro
            if (!$this->model_assignment_group_member->is_member($group_id, $user_id)) {
                $member_data = array(
                    'agm_group_id' => $group_id,
                    'agm_user_id' => $user_id,
                    'agm_added_by' => $this->user_id,
                    'agm_status' => 1
                );
                
                if ($this->model_assignment_group_member->save($member_data)) {
                    $added_count++;
                }
            }
        }
        
        $this->json_param['status'] = true;
        $this->json_param['msg'] = $added_count . ' students added to group';
        echo json_encode($this->json_param);
    }
}
?>
