<?php
class Quiz_group_assignment extends MY_Controller {
    function __construct() {
        parent::__construct();
        $this->load->model('model_quiz');
        $this->load->model('model_quiz_user_assignment');
        $this->load->model('model_assignment_group');
        $this->load->model('model_assignment_group_member');
        
        // Solo administradores y profesores pueden acceder a esta sección
        if (!$this->user_is_admin && $this->user_type != LECTURER_USER) {
            redirect('admin/dashboard/access_denied');
        }
    }
    
    /**
     * Muestra el formulario para asignar un quiz a un grupo
     */
    public function assign_to_group($quiz_id = null) {
        if (!$quiz_id) {
            $this->session->set_flashdata('error', 'Invalid quiz ID');
            redirect('admin/quiz');
        }
        
        // Obtener el quiz
        $quiz = $this->model_quiz->find_by_pk($quiz_id);
        if (!$quiz) {
            $this->session->set_flashdata('error', 'Quiz not found');
            redirect('admin/quiz');
        }
        
        // Procesar el formulario cuando se envía
        if ($this->input->post('submit')) {
            $group_id = $this->input->post('group_id');
            $due_date = $this->input->post('due_date');
            $penalty_rate = $this->input->post('penalty_rate');
            
            // Verificar datos requeridos
            if (!$group_id || !$due_date) {
                $this->session->set_flashdata('error', 'Please select a group and due date');
            } else {
                // Verificar si ya existe esta asignación
                $params = array();
                $params['where']['quiz_id'] = $quiz_id;
                $params['where']['group_id'] = $group_id;
                $existing = $this->model_quiz_user_assignment->find_one($params);
                
                if ($existing) {
                    // Actualizar asignación existente
                    $assignment_data = array(
                        'due_date' => $due_date,
                        'penalty_rate' => $penalty_rate
                    );
                    
                    if ($this->model_quiz_user_assignment->update_by_pk($existing->id, $assignment_data)) {
                        $this->session->set_flashdata('success', 'Quiz assignment updated successfully');
                        redirect('admin/quiz_group_assignment/view_assignments/' . $quiz_id);
                    } else {
                        $this->session->set_flashdata('error', 'Error updating quiz assignment');
                    }
                } else {
                    // Crear nueva asignación
                    $assignment_data = array(
                        'quiz_id' => $quiz_id,
                        'group_id' => $group_id,
                        'user_id' => 0, // 0 indica que es una asignación de grupo, no individual
                        'due_date' => $due_date,
                        'penalty_rate' => $penalty_rate
                    );
                    
                    if ($this->model_quiz_user_assignment->save($assignment_data)) {
                        $this->session->set_flashdata('success', 'Quiz assigned to group successfully');
                        redirect('admin/quiz_group_assignment/view_assignments/' . $quiz_id);
                    } else {
                        $this->session->set_flashdata('error', 'Error assigning quiz to group');
                    }
                }
            }
        }
        
        // Obtener grupos para dropdown (solo grupos de estudiantes, no regulares)
        $groups = $this->model_assignment_group->get_student_groups();
        
        $data['quiz'] = $quiz;
        $data['groups'] = $groups;
        $data['title'] = "Assign Quiz to Group";
        $data['layout_title'] = "Assign Quiz to Group: " . $quiz->title;
        
        $this->load_view("admin/quiz/assign_to_group", $data);
    }
    
    /**
     * Ver todas las asignaciones de un quiz
     */
    public function view_assignments($quiz_id = null) {
        if (!$quiz_id) {
            $this->session->set_flashdata('error', 'Invalid quiz ID');
            redirect('admin/quiz');
        }
        
        // Obtener el quiz
        $quiz = $this->model_quiz->find_by_pk($quiz_id);
        if (!$quiz) {
            $this->session->set_flashdata('error', 'Quiz not found');
            redirect('admin/quiz');
        }
        
        // Obtener todas las asignaciones para este quiz
        $assignments = $this->model_quiz_user_assignment->get_assignments_by_quiz($quiz_id);
        
        $data['quiz'] = $quiz;
        $data['assignments'] = $assignments;
        $data['title'] = "Quiz Assignments";
        $data['layout_title'] = "Assignments for Quiz: " . $quiz->title;
        
        $this->load_view("admin/quiz/view_assignments", $data);
    }
    
    /**
     * Eliminar una asignación
     */
    public function delete_assignment($assignment_id = null, $quiz_id = null) {
        if (!$assignment_id || !$quiz_id) {
            $this->session->set_flashdata('error', 'Invalid parameters');
            redirect('admin/quiz');
        }
        
        if ($this->model_quiz_user_assignment->delete_by_pk($assignment_id)) {
            $this->session->set_flashdata('success', 'Assignment deleted successfully');
        } else {
            $this->session->set_flashdata('error', 'Error deleting assignment');
        }
        
        redirect('admin/quiz_group_assignment/view_assignments/' . $quiz_id);
    }
    
    /**
     * Asignar quiz a grupo (AJAX)
     */
    public function ajax_assign_to_group() {
        // Verificar si es una petición AJAX
        if (!$this->input->is_ajax_request()) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'Invalid request';
            echo json_encode($this->json_param);
            exit;
        }
        
        // Solo profesores y administradores pueden asignar quizzes
        if ($this->user_type != LECTURER_USER && !$this->user_is_admin) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'Permission denied';
            echo json_encode($this->json_param);
            exit;
        }
        
        // Obtener datos
        $quiz_id = $this->input->post('quiz_id');
        $group_id = $this->input->post('group_id');
        $due_date = $this->input->post('due_date');
        $penalty_rate = $this->input->post('penalty_rate');
        
        // Verificar datos
        if (!$quiz_id || !$group_id || !$due_date) {
            $this->json_param['status'] = false;
            $this->json_param['msg'] = 'Missing required fields';
            echo json_encode($this->json_param);
            exit;
        }
        
        // Verificar si ya existe esta asignación
        $params = array();
        $params['where']['quiz_id'] = $quiz_id;
        $params['where']['group_id'] = $group_id;
        $existing = $this->model_quiz_user_assignment->find_one($params);
        
        if ($existing) {
            // Actualizar asignación existente
            $assignment_data = array(
                'due_date' => $due_date,
                'penalty_rate' => $penalty_rate
            );
            
            if ($this->model_quiz_user_assignment->update_by_pk($existing->id, $assignment_data)) {
                $this->json_param['status'] = true;
                $this->json_param['msg'] = 'Quiz assignment updated successfully';
            } else {
                $this->json_param['status'] = false;
                $this->json_param['msg'] = 'Error updating quiz assignment';
            }
        } else {
            // Crear nueva asignación
            $assignment_data = array(
                'quiz_id' => $quiz_id,
                'group_id' => $group_id,
                'user_id' => 0, // 0 indica que es una asignación de grupo, no individual
                'due_date' => $due_date,
                'penalty_rate' => $penalty_rate
            );
            
            if ($this->model_quiz_user_assignment->save($assignment_data)) {
                $this->json_param['status'] = true;
                $this->json_param['msg'] = 'Quiz assigned to group successfully';
            } else {
                $this->json_param['status'] = false;
                $this->json_param['msg'] = 'Error assigning quiz to group';
            }
        }
        
        echo json_encode($this->json_param);
    }
}
?>
