<?php
if (!defined('BASEPATH'))
	exit('No direct script access allowed');

class Quiz_categories extends MY_Controller {

	/**
	 * Controller for managing quiz categories
	 * @package     Quiz Categories
	 * @version     1.0
	 * @since       2025
	 */

    public $_list_data = array();

	public function __construct() {
		global $config;
		
		parent::__construct();

        // Load required models
        $this->load->model('Model_quiz_categories');
        $this->load->model('Model_quiz_category_questions');
        $this->load->model('Model_question_bank');

        // Configure datatable params
        $this->dt_params['dt_headings'] = "qc_id,qc_name,qc_course_id,qc_subject_id,qc_status,question_count";
        $this->dt_params['searchable'] = array("qc_name","qc_course_id","qc_subject_id");
        
        $this->dt_params['action'] = array(
            "hide_add_button" => false,
            "hide" => false,
            "show_delete" => true,
            "show_edit" => true,
            "order_field" => false,
            "show_view" => false,
            "extra" => array(
                'manage_questions_btn' => '<a href="'.la("quiz_categories/manage_questions").'/%d/" class="btn btn-info btn-xs" data-toggle="tooltip" title="Manage Questions"><i class="fa fa-list"></i></a>',
            ),
        );

        // Status options
        $this->_list_data['qc_status'] = array(
            STATUS_ACTIVE => "Active",
            STATUS_INACTIVE => "Inactive",
        );

        // Course and subject lists for dropdowns
        $param = array();
        $this->_list_data['qc_course_id'] = $this->model_course->find_all_list_active($param, 'course_title');

        $param = array();
        $this->_list_data['qc_subject_id'] = $this->model_course_subject->find_all_list_active($param, 'course_subject_name');

        // Process POST data
		$_POST = $this->input->post(NULL, true);
	}

    /**
     * Index page - List all categories
     */
    public function index() {
        // Prepare the data to be passed to view
        $this->add_script(array('quiz_categories.js'), 'js');
        
        // Extend columns definition for datatable to include question count
        $this->extend_dt_params();
        
        // Call parent index function to render the page
        parent::index();
    }

    /**
     * Extend datatable parameters to include question count
     */
    private function extend_dt_params() {
        $this->dt_params['dt_columns'][] = array(
            "db" => "qc_id",
            "dt" => 'question_count',
            "field" => "qc_id",
            "as" => "question_count",
            "callback" => function($value, $data) {
                return $this->Model_quiz_category_questions->count_questions($value);
            }
        );
    }

    /**
     * Get subject list for AJAX dropdown
     */
    public function get_subjects() {
        $course_id = $this->input->post("search_val"); 
        $param['fields'] = "course_subject_id,course_subject_name";
        $param['where']['course_subject_course_id'] = $course_id;
        $data = $this->model_course_subject->find_all_active($param);
        echo json_encode($data);
    }

    /**
     * Add or edit a category
     * @param int $id Category ID
     * @param array $data Form data
     * @return int Inserted/updated ID
     */
    public function add($id = 0, $data = array()) {
        // Add required JS files
        $this->add_script(array('quiz_categories.js'), 'js');
        
        // Call parent add method to handle the form
        return parent::add($id, $data);
    }

    /**
     * Manage questions for a category
     * @param int $category_id Category ID
     */
    public function manage_questions($category_id = 0) {
        // Check if category exists
        $category = $this->Model_quiz_categories->get_data(array(), $category_id);
        if (!$category) {
            redirect('admin/quiz_categories');
        }
        
        // Add necessary scripts and plugins
        $this->add_script(array('quiz_categories.js'), 'js');
        $this->register_plugins(array(
    "bootstrap",
    "datatables",
    "flag_sprites",
    "jquery-ui",
    "bootstrap-hover-dropdown",
    "jquery-slimscroll",
    "uniform",
    "bootstrap-switch",
    "font-awesome",
    "simple-line-icons",
    "select2",
    "bootbox",
    "bootstrap-toastr"
));

        // Get all questions in this category
        $category_questions = $this->Model_quiz_category_questions->get_category_questions($category_id);
        
        // Get the course and subject info for filtering available questions
        $course_id = $category['qc_course_id'];
        $subject_id = $category['qc_subject_id'];
        
        // Set up data for the view
        $data = array(
            'category' => $category,
            'category_questions' => $category_questions,
            'course_id' => $course_id,
            'subject_id' => $subject_id
        );
        
        // Load the view
        $this->load_view('manage_questions', $data);
    }

    /**
     * AJAX endpoint to get questions for a category
     */
    public function get_category_questions() {
        $category_id = $this->input->post('category_id');
        if (!$category_id) {
            echo json_encode(array('status' => false, 'message' => 'Invalid category ID'));
            return;
        }
        
        $questions = $this->Model_quiz_category_questions->get_category_questions($category_id);
        echo json_encode(array('status' => true, 'questions' => $questions));
    }

    /**
     * AJAX endpoint to add questions to a category
     */
    public function add_questions() {
        $category_id = $this->input->post('category_id');
        $question_ids = $this->input->post('question_ids');
        
        if (!$category_id || !$question_ids) {
            echo json_encode(array('status' => false, 'message' => 'Invalid data provided'));
            return;
        }
        
        // Convert question_ids to array if it's a string
        if (!is_array($question_ids)) {
            $question_ids = explode(',', $question_ids);
        }
        
        // Add questions to category
        $result = $this->Model_quiz_category_questions->assign_multiple_questions($category_id, $question_ids);
        
        echo json_encode(array(
            'status' => true,
            'result' => $result,
            'message' => sprintf(
                '%d questions added, %d duplicates, %d errors', 
                $result['success'], 
                $result['duplicates'], 
                $result['error']
            )
        ));
    }

    /**
     * AJAX endpoint to remove a question from a category
     */
    public function remove_question() {
        $category_id = $this->input->post('category_id');
        $question_id = $this->input->post('question_id');
        
        if (!$category_id || !$question_id) {
            echo json_encode(array('status' => false, 'message' => 'Invalid data provided'));
            return;
        }
        
        $result = $this->Model_quiz_category_questions->remove_question($category_id, $question_id);
        echo json_encode(array('status' => $result));
    }
}

/* End of file Quiz_categories.php */
/* Location: ./application/controllers/admin/Quiz_categories.php */
