<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Question_bank extends MY_Controller {

    /**
     * Centralized Question Bank
     * @package        Question_bank
     * @version        1.0
     * @since          Version 1.0 2025
     */

    public $_list_data = array();

    public function __construct() {
        global $config;
        parent::__construct();

        // Load required models
        $this->load->model("model_question_bank");
        $this->load->model("model_quiz_questions");
        $this->load->model("model_course");
        $this->load->model("model_course_subject");
        $this->load->model("model_quiz");

        // Standard DataTables configuration
        // Updated to match actual columns: checkbox,id,question,course,quizzes,subject,type,base_points,is_bonus,status,actions
        $this->dt_params['dt_headings'] = "checkbox,qb_id,qb_text,qb_course_id,assigned_quizzes,qb_subject_id,qb_type,qb_gain_point,qb_is_bonus,qb_status,actions";
        $this->dt_params['searchable'] = explode(",", $this->dt_params['dt_headings']);
        $this->dt_params['action'] = array(
            "hide_add_button" => false,
            "hide" => false,
            "show_delete" => true,
            "show_edit" => true,
            "order_field" => false,
            "show_view" => false,
            "extra" => array(
                // Action of assign to quiz deleted for cleaning and separation of responsibilities
            ),
        );

        // Available statuses for questions
        $this->_list_data['qb_status'] = array(
            STATUS_INACTIVE => "<span class=\"label label-default\">Inactive</span>",
            STATUS_ACTIVE =>  "<span class=\"label label-primary\">Active</span>"
        );

        // Question types
        $this->_list_data['qb_type'] = array(
            1 => "Multiple Choice",
            2 => "True/False",
            3 => "Essay",
            4 => "Short Answer",
            5 => "Fill in the Blank"
        );

        // Courses and subjects available for filtering
        $param = array('where' => array('course_status' => 1));
        $this->_list_data['qb_course_id'] = $this->model_course->find_all_list_active($param, 'course_title');
        $param = array();
        $this->_list_data['qb_subject_id'] = $this->model_course_subject->find_all_list_active($param,'course_subject_name');

        // For use in JS files
        $config['js_config']['paginate'] = isset($this->dt_params['paginate']) ? $this->dt_params['paginate'] : array();

        // Register base plugins in the constructor
        $this->register_plugins(array("bootstrap", "datatables", "flag_sprites", "jquery-ui"));
        // Add question bank JS script


        // Normalize $_POST for compatibility
        $_POST = $this->input->post(NULL, true);
    }
    
    /**
     * Main view for the centralized question bank
     */
    public function index() {
        // Título de la página
        $this->layout_data['template_title'] = "Question Bank";
        $this->layout_data['page_title'] = 'Question Bank';

        // Register plugins same as quiz_bank
        $this->register_plugins(array(
            "jquery-ui",
            "bootstrap",
            "bootstrap-hover-dropdown",
            "jquery-slimscroll",
            "uniform",
            "bootstrap-switch",
            "bootstrap-datepicker",
            "font-awesome",
            "simple-line-icons",
            "select2",
            "datatables",
            "bootbox",
            "bootstrap-toastr",
            "flag_sprites"
        ));

        // Assign plugins to additional_tools (NOT the view)
        $this->layout_data['additional_tools'] = array(
            "jquery-ui",
            "bootstrap",
            "bootstrap-hover-dropdown",
            "jquery-slimscroll",
            "uniform",
            "bootstrap-switch",
            "bootstrap-datepicker",
            "font-awesome",
            "simple-line-icons",
            "select2",
            "datatables",
            "bootbox",
            "bootstrap-toastr",
            "flag_sprites"
        );
        // If you need the filter view, assign it to another variable:
        $this->layout_data['filters_html'] = $this->load->view("admin/question_bank/partials/additional_tools", array(), true);

        // Custom JS files required for Metronic-style DataTable
        $this->layout_data['js_files'] = array(
            'datatable.js', // Wrapper Datatable
            'question_bank.js' // Inicializador específico del banco de preguntas
        );

        // Prepare data for the main view
        $this->layout_data['data'] = null;
        $this->layout_data['form_action'] = "";
        
        // Load the main view - using relative path without prefix
        parent::load_view("index");
    }
    
    /**
     * Gets data for DataTables with support for pagination, filtering, and search
     */
    public function paginate($dt_params = array()) {
        // Uses the same logic as get_question_bank_data
        return $this->get_question_bank_data();
    }

    public function get_question_bank_data() {
        // In CodeIgniter 3, properties like input, load, etc. are initialized via the parent class. Ensure we use the properties correctly.
        
        // Verify if it is an AJAX request
        if (!isset($_SERVER['HTTP_X_REQUESTED_WITH']) || strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) != 'xmlhttprequest') {
            echo json_encode(array('status' => false, 'message' => 'Invalid request'));
            return;
        }
        
        // DEBUG LOGS
        log_message('error', 'POST: ' . print_r($_POST, true));
        log_message('error', 'SESSION: ' . print_r($this->session->userdata(), true));
        
        try {
            // Process DataTables request parameters using $_POST directly
            $draw = isset($_POST['draw']) ? intval($_POST['draw']) : 1;
            $start = isset($_POST['start']) ? intval($_POST['start']) : 0;
            $length = isset($_POST['length']) ? intval($_POST['length']) : 10;
            $search = isset($_POST['search']['value']) ? $_POST['search']['value'] : '';
            
            // Get additional filters
            $course_id = isset($_POST['course_id']) ? $_POST['course_id'] : null;
            $subject_id = $this->input->post('subject_id');
            $type = $this->input->post('question_type');
            $show_only_unassigned = $this->input->post('show_only_unassigned');
            
            // Default values
            if ($length < 1) $length = 10;
            
            // Parameters for filtering
            $params = array(
                'course_id' => $course_id,
                'subject_id' => $subject_id,
                'type' => $type,
                'search' => $search,
                'status' => 1 // Only active questions
            );
            
            // Pagination settings
            $pagination = array(
                'start' => $start,
                'length' => $length
            );
            
            // Get total records without filtering
            $recordsTotal = $this->model_question_bank->count_questions(array('status' => 1));
            
            // Get total filtered records
            $recordsFiltered = $this->model_question_bank->count_questions($params);
            
            // Get filtered and paginated questions
            $questions = $this->model_question_bank->get_questions($params, $pagination);
            
            // Prepare data for DataTables (array of arrays, not associative)
            $data = array();
            foreach ($questions as $question) {
                // Get usage information (in which quizzes it is assigned)
                $usage = $this->model_question_bank->get_question_usage($question['qb_id']);
                
                // If only unassigned questions are requested and this question is assigned, skip it
                if ($show_only_unassigned && !empty($usage)) {
                    continue;
                }
                
                // Process assignments (fixed: initialize arrays for each question)
                $assigned_courses = array();
                $assigned_quizzes = array();
                log_message('error', 'DEBUG USAGE qb_id '.$question['qb_id'].': '.print_r($usage, true));
                if (!empty($usage)) {
                    foreach ($usage as $quiz) {
                        log_message('error', 'DEBUG USAGE ITEM: '.print_r($quiz, true));
                        if (!empty($quiz['course_title'])) {
                            $assigned_courses[$quiz['course_title']] = true;
                        }
                        if (!empty($quiz['quiz_name'])) {
                            $assigned_quizzes[] = $quiz['quiz_name'];
                        }
                    }
                }
                $assigned_courses_str = $assigned_courses ? implode(', ', array_keys($assigned_courses)) : '-';
                $assigned_quizzes_str = $assigned_quizzes ? implode(', ', $assigned_quizzes) : '-';

                // Get subject name from the list loaded in constructor
                $subject_id_int = isset($question['qb_subject_id']) ? (int)$question['qb_subject_id'] : 0;
                $subject_name = ($subject_id_int && isset($this->_list_data['qb_subject_id'][$subject_id_int]))
                    ? $this->_list_data['qb_subject_id'][$question['qb_subject_id']]
                    : 'No subject';

                // Prepare data for each row (in the same order as the <th> in the view)
                $row = array();
                // 0: Checkbox
                $row[] = '<input type="checkbox" class="question-checkbox" value="'.$question['qb_id'].'"/>';
                // 1: ID
                $row[] = $question['qb_id'];
                // 2: Question
                $row[] = substr(strip_tags($question['qb_text']), 0, 100) . (strlen($question['qb_text']) > 100 ? '...' : '');
                // 3: Question's course (not the assigned ones)
                $course_name = ($course_id_int = isset($question['qb_course_id']) ? (int)$question['qb_course_id'] : 0) && isset($this->_list_data['qb_course_id'][$course_id_int])
                    ? $this->_list_data['qb_course_id'][$course_id_int]
                    : 'No course';
                $row[] = $course_name;
                // 4: Assigned quizzes
                $row[] = $assigned_quizzes_str;
                // 5: Subject
                $row[] = $subject_name;
                // 6: Type
                $type_id_int = isset($question['qb_type']) ? (int)$question['qb_type'] : 0;
                $row[] = ($type_id_int && isset($this->_list_data['qb_type'][$type_id_int]))
                    ? $this->_list_data['qb_type'][$question['qb_type']]
                    : 'Unknown';
                // 7: Base Points
                $base_points = isset($question['qb_gain_point']) && !empty($question['qb_gain_point']) ? $question['qb_gain_point'] : '1.0';
                $row[] = '<span class="label label-info">' . $base_points . ' pts</span>';
                // 8: Is bonus?
                $row[] = $question['qb_is_bonus'] ? '<span class="label label-success">Yes</span>' : 'No';
                // 9: Status
                $row[] = $this->_list_data['qb_status'][$question['qb_status']];
                // 10: Actions
                $actions = '<div class="action-btn-container">';
                $actions .= '<a href="'.site_url('admin/question_bank/edit/'.$question['qb_id']).'" class="btn btn-xs btn-primary" title="Edit"><i class="fa fa-pencil"></i></a> ';
                $actions .= '<a href="javascript:void(0);" class="btn btn-xs btn-danger delete-question-btn" data-id="'.$question['qb_id'].'" title="Delete"><i class="fa fa-trash"></i></a> ';
                // Assign to quiz action removed for clarity and separation of responsibilities
                $actions .= '</div>';
                $row[] = $actions;

                $data[] = $row;
            }
            
            // --- FINAL RESPONSE FOR DATATABLES ---
            $response = array(
                "draw" => $draw,
                "recordsTotal" => $recordsTotal,
                "recordsFiltered" => $recordsFiltered,
                "data" => $data
            );
            header('Content-Type: application/json');
            echo json_encode($response);
            return;
        } catch (Exception $e) {
            // Error response
            echo json_encode(array(
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => array(),
                'error' => $e->getMessage()
            ));
        }
    }
    

    public function edit($id) {
        // ID verification
        if (empty($id)) {
            log_message('error', 'The $id parameter did not reach the edit() method of Question_bank');
            show_error('No valid ID was provided to edit the question.');
        } else {
            log_message('debug', 'ID received in edit(): ' . $id);
        }

        $question = $this->model_question_bank->find_by_pk($id);
        if (!$question) {
            $this->session->set_flashdata('error', 'Question not found.');
            redirect('admin/question_bank');
        }
        $this->layout_data['template_title'] = "Edit Question";
        $this->layout_data['page_title'] = 'Edit Question';
        if ($this->input->post()) {
            log_message('error', '[EDIT] POST received: ' . print_r($this->input->post(), true));
            $this->form_validation->set_rules('question[qb_text]', 'Question Text', 'required');
            $this->form_validation->set_rules('question[qb_type]', 'Question Type', 'required|numeric');
            $this->form_validation->set_rules('question[qb_course_id]', 'Course', 'required|numeric');
            $this->form_validation->set_rules('question[qb_subject_id]', 'Subject', 'required|numeric');
            $this->form_validation->set_rules('question[qb_gain_point]', 'Base Points', 'required|numeric|greater_than[0]');
            $valid = $this->form_validation->run();
            log_message('error', '[EDIT] Validation result: ' . ($valid ? 'OK' : 'FAILED'));
            if ($valid) {
                $question_data = $this->input->post('question');
                if (isset($_POST['options']) && !empty($_POST['options'])) {
                    $question_data['qb_options'] = json_encode($_POST['options']);
                }
                log_message('error', '[EDIT] Data sent to save_question: ' . print_r($question_data, true));
                $result = $this->model_question_bank->save_question($question_data, $id);
                log_message('error', '[EDIT] Result of save_question: ' . print_r($result, true));
                if ($result) {
                    $this->session->set_flashdata('success', 'Question updated successfully.');
                    redirect('admin/question_bank');
                } else {
                    $this->session->set_flashdata('error', 'Failed to update question. Please try again.');
                }
            } else {
                $this->session->set_flashdata('error', validation_errors());
            }
        }
        if (isset($question['qb_options']) && !empty($question['qb_options'])) {
            $options = json_decode($question['qb_options'], true);
            // Normalize: each option must be an associative array
            if (is_array($options)) {
                foreach ($options as $i => $opt) {
                    if (!is_array($opt)) {
                        $options[$i] = ['text' => $opt];
                    }
                }
            }
            $question['options'] = $options;
        }
        $courses_result = $this->model_course->find_all_list_active(array('where' => array('course_status' => 1)), 'course_title');

        $this->layout_data['courses'] = $courses_result;

        $this->layout_data['subjects'] = $this->model_course_subject->find_all_list_active(array(), 'course_subject_name');
        $this->layout_data['question_types'] = $this->_list_data['qb_type'];
        $this->layout_data['question'] = $question;
        $this->layout_data['usage'] = $this->model_question_bank->get_question_usage($id);
        // Register plugins and visual scripts the same as in index
        $this->register_plugins(array(
            "bootstrap",
            "datatables",
            "flag_sprites",
            "jquery-ui",
            "bootstrap-hover-dropdown",
            "jquery-slimscroll",
            "uniform",
            "bootstrap-switch",
            "font-awesome",
            "simple-line-icons",
            "select2",
            "bootbox",
            "bootstrap-toastr"
        ));
        $this->add_script(array(
            "metronic.js",
            "layout.js",
            "quick-sidebar.js",
            "demo.js",
            "table-ajax.js",
            "ui-alert-dialog-api.js"
        ), "js");
        $this->load_view('form', $this->layout_data);
    }
    
    /**
     * Delete a question from the question bank (AJAX)
     */
    public function delete_question() {
        // Only accept AJAX POST requests
        if (!$this->input->is_ajax_request() || !$this->input->post('question_id')) {
            echo json_encode(array('status' => false, 'message' => 'Invalid request.'));
            return;
        }
        $question_id = $this->input->post('question_id');
        $result = $this->model_question_bank->delete_question($question_id);
        if ($result) {
            echo json_encode(array('status' => true, 'message' => 'Question deleted successfully.'));
        } else {
            echo json_encode(array('status' => false, 'message' => 'Failed to delete question.'));
        }
    }
    
    /**
     * Get questions that are available to be added to a category
     * Used by the quiz category management
     */
    public function get_available_questions_json()
    {
        // Enable error reporting for debugging
        ini_set('display_errors', 1);
        error_reporting(E_ALL);
        
        log_message('error', 'INICIO get_available_questions_json - VERSION ROBUSTA');
        
        try {
            // Load required models
            $this->load->model('Model_quiz_category_questions');
            
            // Get parameters
            $draw = intval($this->input->post('draw'));
            $start = intval($this->input->post('start'));
            $length = intval($this->input->post('length'));
            $search = isset($this->input->post('search')['value']) ? $this->input->post('search')['value'] : '';
            $category_id = intval($this->input->post('category_id'));
            
            log_message('error', 'Params - draw: ' . $draw . ', start: ' . $start . ', length: ' . $length . ', category_id: ' . $category_id);
            
            // Set up filters
            $params = array();
            $params['offset'] = $start;
            $params['limit'] = $length;
            
            // Apply search filter
            if (!empty($search)) {
                $params['where_like'][] = array('field' => 'qb_text', 'value' => $search, 'position' => 'both');
            }
            
            // Apply course filter
            $course_id = intval($this->input->post('course_id'));
            if (!empty($course_id)) {
                $params['where']['qb_course_id'] = $course_id;
            }
            
            // Apply subject filter
            $subject_id = intval($this->input->post('subject_id'));
            if (!empty($subject_id)) {
                $params['where']['qb_subject_id'] = $subject_id;
            }
            
            // Try to exclude questions already in category
            try {
                if (!empty($category_id)) {
                    // Get questions from the category
                    $excluded_questions = $this->Model_quiz_category_questions->get_category_questions($category_id);
                    
                    // Exclude questions if any
                    if (!empty($excluded_questions)) {
                        $exclude_ids = array_column($excluded_questions, 'qb_id');
                        if (!empty($exclude_ids)) {
                            $params['where_not_in'] = array('field' => 'qb_id', 'values' => $exclude_ids);
                        }
                    }
                }
            } catch (Exception $ex) {
                // Continue even if there is an error with the exclusion
                log_message('error', 'Error al obtener preguntas excluidas: ' . $ex->getMessage());
            }
            
            // Use get_questions from the model
            $question_params = array();
            $pagination = array('start' => $start, 'length' => $length);
            
            if (!empty($search)) {
                $question_params['search'] = $search;
            }
            if (!empty($course_id)) {
                $question_params['course_id'] = $course_id;
            }
            if (!empty($subject_id)) {
                $question_params['subject_id'] = $subject_id;
            }
            // You can add more filters according to the model
            
            // Exclude questions already in category
            if (!empty($category_id)) {
                $excluded_questions = $this->Model_quiz_category_questions->get_category_questions($category_id);
                if (!empty($excluded_questions)) {
                    $exclude_ids = array_column($excluded_questions, 'qb_id');
                    if (!empty($exclude_ids)) {
                        $question_params['where_not_in'] = array('qb_id' => $exclude_ids);
                    }
                }
            }
            
            // Get total records without filter
            $total_records = $this->model_question_bank->get_questions(array(), null);
            $total_records = is_array($total_records) ? count($total_records) : 0;
            
            // Get filtered and paginated records
            $questions = $this->model_question_bank->get_questions($question_params, $pagination);
            $filtered_records = is_array($questions) ? count($questions) : 0;

            // LOG: Ver el resultado de $questions
            log_message('error', 'SUCCESS: RESULTADO $questions: ' . print_r($questions, true));
            
            // If you want to show the course/subject name, search for it here:
            foreach ($questions as &$q) {
                // Get course name
                if (isset($q['qb_course_id']) && $q['qb_course_id']) {
                    $course = $this->model_course->find_by_pk($q['qb_course_id']);
                    $q['course_title'] = isset($course['course_title']) ? $course['course_title'] : '';
                } else {
                    $q['course_title'] = '';
                }
                // Get subject name
                if (isset($q['qb_subject_id']) && $q['qb_subject_id']) {
                    $subject = $this->model_course_subject->find_by_pk($q['qb_subject_id']);
                    $q['subject_name'] = isset($subject['course_subject_name']) ? $subject['course_subject_name'] : '';
                } else {
                    $q['subject_name'] = '';
                }
            }
            unset($q);

            // Format for DataTables (expected fields)
            $data = array();
            foreach ($questions as $question) {
                $question_text = isset($question['qb_text']) ? strip_tags($question['qb_text']) : '';
                if (strlen($question_text) > 100) {
                    $question_text = substr($question_text, 0, 100) . '...';
                }
                $data[] = array(
                    'qb_id' => isset($question['qb_id']) ? $question['qb_id'] : '',
                    'qb_text' => $question_text,
                    // Map type ID to human-readable label, fallback to "Unknown"
                    'type' => isset($question['qb_type']) && isset($this->_list_data['qb_type'][$question['qb_type']])
                        ? $this->_list_data['qb_type'][$question['qb_type']]
                        : 'Unknown',
                    // Course and subject fallback strings if empty
                    'course_title' => !empty($question['course_title']) ? $question['course_title'] : 'No course',
                    'subject_name' => !empty($question['subject_name']) ? $question['subject_name'] : 'No subject'
                );
            }
            
            // Get total records (without filter)
            $total_records = 0;
            try {
                $total_records = $this->model_question_bank->find_count();
                log_message('error', 'SUCCESS: Total records: ' . $total_records);
            } catch (Exception $ex) {
                log_message('error', 'ERROR: Error al obtener total records: ' . $ex->getMessage());
            }
            
            // Get filtered records
            $filtered_records = 0;
            $questions = array();
            try {
                // Try count query
                $filter_params = $params;
                $filter_params['return_count'] = true;
                $filtered_result = $this->model_question_bank->find_all($filter_params);
                
                if (isset($filtered_result['count'])) {
                    $filtered_records = $filtered_result['count'];
                    $questions = isset($filtered_result['data']) ? $filtered_result['data'] : array();
                } else {
                    // Fallback if no count is returned
                    $questions = $this->model_question_bank->find_all($params);
                    $filtered_records = count($questions);
                }
                
                log_message('error', 'SUCCESS: Filtered records: ' . $filtered_records . ', Questions count: ' . count($questions));
            } catch (Exception $ex) {
                log_message('error', 'ERROR: Error en consulta de registros filtrados: ' . $ex->getMessage());
            }
            
            // Format data for DataTables
            $data = array();
            foreach ($questions as $question) {
                try {
                    // Complete course and subject titles if missing
                    if (!isset($question['course_title'])) {
                        if (!empty($question['qb_course_id'])) {
                            $courseObj = $this->model_course->find_by_pk($question['qb_course_id']);
                            $question['course_title'] = $courseObj && isset($courseObj['course_title']) ? $courseObj['course_title'] : '';
                        } else {
                            $question['course_title'] = '';
                        }
                    }
                    if (!isset($question['subject_name'])) {
                        if (!empty($question['qb_subject_id'])) {
                            $subjectObj = $this->model_course_subject->find_by_pk($question['qb_subject_id']);
                            $question['subject_name'] = $subjectObj && isset($subjectObj['course_subject_name']) ? $subjectObj['course_subject_name'] : '';
                        } else {
                            $question['subject_name'] = '';
                        }
                    }
                    // Process question text to remove HTML and limit length
                    $question_text = isset($question['qb_text']) ? strip_tags($question['qb_text']) : 'No text';
                    if (strlen($question_text) > 100) {
                        $question_text = substr($question_text, 0, 100) . '...';
                    }
                    
                    $data[] = array(
                        'qb_id' => isset($question['qb_id']) ? $question['qb_id'] : 0,
                        'qb_text' => $question_text,
                        'type' => isset($question['qb_type']) && isset($this->_list_data['qb_type'][$question['qb_type']])
                            ? $this->_list_data['qb_type'][$question['qb_type']]
                            : (isset($question['type']) ? $question['type'] : 'Unknown'),
                        'course_title' => isset($question['course_title']) ? $question['course_title'] : 'No course',
                        'subject_name' => isset($question['subject_name']) ? $question['subject_name'] : 'No subject'
                    );
                } catch (Exception $ex) {
                    log_message('error', 'ERROR: Error al formatear pregunta: ' . $ex->getMessage());
                }
            }
            
            // Return JSON for DataTables
            $output = array(
                "draw" => $draw,
                "recordsTotal" => $total_records,
                "recordsFiltered" => $filtered_records,
                "data" => $data
            );
            
            log_message('error', 'SUCCESS: Success response with ' . count($data) . ' questions');
            echo json_encode($output);
            exit;
            
        } catch (Exception $e) {
            log_message('error', 'ERROR: General ERROR in get_available_questions_json: ' . $e->getMessage());
            
            // Respond with a formatted error for DataTables
            $draw = intval($this->input->post('draw'));
            echo json_encode(array(
                "draw" => $draw,
                "recordsTotal" => 0,
                "recordsFiltered" => 0,
                "data" => array(),
                "error" => 'Server error: ' . $e->getMessage()
            ));
            exit;
        }
    }
    
/**
 * Crear una nueva pregunta
 */
public function create() {
    // Debug: log POST completo para ver si llega quiz_id
    log_message('error', '[DEBUG] POST en create: ' . print_r($_POST, true));
    $this->layout_data['template_title'] = "Create Question";
        $this->layout_data['page_title'] = 'Create Question';

        if (
            $this->input->post()
        ) {
        $this->form_validation->set_rules('question[qb_text]', 'Question Text', 'required');
        $this->form_validation->set_rules('question[qb_type]', 'Question Type', 'required|numeric');
        $this->form_validation->set_rules('question[qb_course_id]', 'Course', 'required|numeric');
        $this->form_validation->set_rules('question[qb_subject_id]', 'Subject', 'required|numeric');
        $this->form_validation->set_rules('question[qb_gain_point]', 'Base Points', 'required|numeric|greater_than[0]');

        if ($this->form_validation->run()) {
            $question_data = $this->input->post('question');
            $admin_id = $this->session->userdata('admin_id');
            if (empty($admin_id)) {
              log_message('error', 'admin_id not found in session. Using 1 by default.');
                $admin_id = 1; // O el ID del super admin
            }
            $question_data['qb_created_by'] = $admin_id;
            if (isset($_POST['options']) && !empty($_POST['options'])) {
                $question_data['qb_options'] = json_encode($_POST['options']);
            }
            $question_id = $this->model_question_bank->save_question($question_data);
            if ($question_id) {
                // Si se recibe quiz_id en el POST, asignar la pregunta al quiz
                $quiz_id = $this->input->post('quiz_id');
                if (!empty($quiz_id)) {
                    $assigned_id = $this->model_quiz_questions->assign_question($quiz_id, $question_id);
                    if (!$assigned_id) {
                        log_message('error', 'ERROR: Could not assign question to quiz automatically.');
                        $this->session->set_flashdata('warning', 'Question created but could not be assigned to the quiz.');
                    }
                }
                $this->session->set_flashdata('success', 'Question created successfully.');
                redirect('admin/question_bank');
            } else {
                $this->session->set_flashdata('error', 'Failed to create question. Please try again.');
            }
        } else {
            $this->session->set_flashdata('error', validation_errors());
        }
    }
    log_message('error', 'PRUEBA DE LOG');
    // Register plugins and visual scripts like in index
    $this->register_plugins(array(
        "bootstrap",
        "datatables",
        "flag_sprites",
        "jquery-ui",
        "bootstrap-hover-dropdown",
        "jquery-slimscroll",
        "uniform",
        "bootstrap-switch",
        "font-awesome",
        "simple-line-icons",
        "select2",
        "bootbox",
        "bootstrap-toastr"
    ));
    $this->add_script(array(
        "metronic.js",
        "layout.js",
        "quick-sidebar.js",
        "demo.js",
        "table-ajax.js",
        "ui-alert-dialog-api.js"
    ), "js");
    $this->layout_data['courses'] = $this->model_course->find_all_list_active(array('where' => array('course_status' => 1)), 'course_title');
    $this->layout_data['subjects'] = $this->model_course_subject->find_all_list_active(array(), 'course_subject_name');
    $this->layout_data['question_types'] = (isset($this->_list_data['qb_type']) && is_array($this->_list_data['qb_type'])) ? $this->_list_data['qb_type'] : array();
    log_message('debug', 'DEBUG courses: ' . print_r($this->layout_data['courses'], true));
    extract($this->layout_data);
    $this->load_view('form', $this->layout_data);
}

/**
 * Assign selected questions to a quiz (AJAX)
 */
public function assign_questions() {
    // Only accept AJAX POST requests
    if (!$this->input->is_ajax_request() || !$this->input->post()) {
        echo json_encode(array('status' => false, 'message' => 'Invalid request.'));
        return;
    }
    // Log received data for debugging
    log_message('error', 'assign_questions POST: ' . print_r($this->input->post(), true));

    $quiz_id = $this->input->post('quiz_id');
    $question_ids = $this->input->post('question_ids');

    // If question_ids comes as a JSON string, decode it
    if (is_string($question_ids)) {
        $decoded = json_decode($question_ids, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            $question_ids = $decoded;
        }
    }

    // Validate data
    if (empty($quiz_id) || empty($question_ids) || !is_array($question_ids)) {
        echo json_encode(array('status' => false, 'message' => 'Missing or invalid parameters.'));
        return;
    }

    // Assign questions
    $result = $this->model_quiz_questions->assign_multiple_questions($quiz_id, $question_ids);

    // Prepare response
    $response = array(
        'status' => ($result['success'] > 0),
        'success_count' => $result['success'],
        'error_count' => $result['error'],
        'duplicate_count' => $result['duplicates'],
        'message' => sprintf(
            'Assigned %d questions to the quiz. %d errors. %d already assigned.',
            $result['success'], $result['error'], $result['duplicates']
        )
    );

    echo json_encode($response);
}

    /**
     * Get available quizzes for assignment
     */
    public function get_quizzes() {
        // Verify if it's an AJAX request
        if (!$this->input->is_ajax_request()) {
            echo json_encode(array('status' => false, 'message' => 'Invalid request'));
            return;
        }
        
        // Get active quizzes
        $params = array('where' => array('quiz_status' => 1));
        $quizzes = $this->model_quiz->find_all($params);
        
        // Prepare response
        $response = array(
            'status' => true,
            'quizzes' => array()
        );
        
        foreach ($quizzes as $quiz) {
            $response['quizzes'][] = array(
                'id' => $quiz['quiz_id'],
                'name' => $quiz['quiz_name'],
                'course' => $this->model_course->find_by_pk($quiz['quiz_course_id'])['course_title'] ?? '',
                'subject' => $quiz['quiz_subject_id'] ? 'Subject '.$quiz['quiz_subject_id'] : ''
            );
        }
        
        echo json_encode($response);
    }
    
    /**
     * Get subjects based on a course
     * @param int $course_id Course ID
     */
    public function get_subjects($course_id = null) {
        // Verify if it's an AJAX request
        if (!$this->input->is_ajax_request()) {
            echo json_encode(array('status' => false, 'message' => 'Invalid request'));
            return;
        }
        // If no course_id is provided, try to get it from POST
        if ($course_id === null) {
            $course_id = $this->input->post('course_id');
        }
        // Validate course_id
        if (empty($course_id)) {
            echo json_encode(array('status' => false, 'message' => 'Course ID is required.'));
            return;
        }
        // Get subjects of the course
        $this->load->model('model_course_subject');
        $params = array('where' => array('cs_course_id' => $course_id, 'cs_status' => 1));
        $subjects = $this->model_course_subject->find_all($params);
        // Prepare response
        $response = array(
            'status' => true,
            'subjects' => array()
        );
        foreach ($subjects as $subject) {
            $response['subjects'][] = array(
                'id' => $subject['cs_id'],
                'name' => $subject['cs_title']
            );
        }
        echo json_encode($response);
    }
    
    /**
     * Returns available question types as JSON for AJAX requests
     */
    public function get_types() {
        if (!$this->input->is_ajax_request()) {
            echo json_encode(['status' => false, 'message' => 'Invalid request']);
            return;
        }
        $types = [
            ['id' => 1, 'name' => 'Multiple Choice'],
            ['id' => 2, 'name' => 'True/False'],
            ['id' => 3, 'name' => 'Essay'],
            ['id' => 4, 'name' => 'Short Answer'],
            ['id' => 5, 'name' => 'Fill in the Blank'],
        ];
        echo json_encode(['status' => true, 'types' => $types]);
    }

    /**
     * Returns active courses as JSON for AJAX requests
     */
    public function get_courses() {
        if (!$this->input->is_ajax_request()) {
            echo json_encode(['status' => false, 'message' => 'Invalid request']);
            return;
        }
        $this->load->model('model_course');
        $params = ['where' => ['course_status' => 1]];
        $courses = $this->model_course->find_all($params);
        $result = [];
        foreach ($courses as $course) {
            $result[] = [
                'id' => $course['course_id'],
                'name' => $course['course_title']
            ];
        }
        echo json_encode(['status' => true, 'courses' => $result]);
    }

    public function import() {
        // Title of the page
        $this->layout_data['template_title'] = "Import Questions";
        
        // Load import view - using relative path without prefix
        parent::load_view("import");
    }
    
    /**
     * Process import file
     */
    public function process_import() {
        // Implement logic to process Excel file
        // [Note: This method should be implemented according to specific requirements]
    }
}
