<?php if (!defined('BASEPATH')) exit('No direct script access allowed');

class Migrate_questions extends MY_Controller {

    /**
     * Script para migrar preguntas del sistema antiguo al nuevo banco centralizado
     * @package        Migrate_questions
     * @version        1.0
     * @since          Version 1.0 2025
     */

    public function __construct() {
        parent::__construct();
        
        // Solo verificar sesión si no es llamada CLI
        if (php_sapi_name() !== 'cli' && !$this->session->userdata('admin_id')) {
            redirect('admin/login');
        }
        
        // Cargar modelos necesarios
        $this->load->model('model_question_bank');
        $this->load->model('model_quiz_questions');
        
        // Deshabilitar layout para esta página
        $this->layout = false;
    }
    
    /**
     * Migrar todas las preguntas del sistema antiguo al nuevo
     * Esta funciu00f3n debe ejecutarse una sola vez
     */
    public function index() {
        // Verificar si las tablas del nuevo sistema existen
        if (!$this->db->table_exists('question_bank') || !$this->db->table_exists('quiz_questions')) {
            echo '<h1>Error: Las tablas del banco de preguntas centralizado no existen</h1>';
            echo '<p>Ejecute las migraciones primero</p>';
            return;
        }
        
        // Iniciar contador de tiempo para mediciu00f3n de rendimiento
        $start_time = microtime(true);
        
        // Contadores para estadisticas
        $stats = [
            'total_questions' => 0,
            'migrated_questions' => 0,
            'failed_questions' => 0,
            'total_relations' => 0,
            'migrated_relations' => 0,
            'failed_relations' => 0,
            'unassigned_questions' => 0,
            'assigned_questions' => 0
        ];
        
        // Obtener todas las preguntas existentes de tn_course_quiz
        $this->db->select('*');
        $existing_questions = $this->db->get('tn_course_quiz')->result_array();
        $stats['total_questions'] = count($existing_questions);
        
        echo '<h1>Migración de Preguntas</h1>';
        echo '<p>Iniciando migración de ' . $stats['total_questions'] . ' preguntas...</p>';
        
        // Mapeo de IDs antiguos a nuevos para mantener relaciones
        $id_mapping = [];
        
        // Procesar cada pregunta
        foreach ($existing_questions as $question) {
            try {
                // Preparar datos para el nuevo formato
                $new_question = [
                    'qb_text' => $question['cq_question'],
                    'qb_type' => $question['cq_type'],
                    'qb_is_bonus' => isset($question['cq_is_bonus']) ? $question['cq_is_bonus'] : 0,
                    'qb_bonus_points' => isset($question['cq_bonus_points']) ? $question['cq_bonus_points'] : 0,
                    'qb_difficulty' => 3, // Dificultad media por defecto
                    'qb_course_id' => $question['cq_course_id'],
                    'qb_subject_id' => isset($question['cq_subject_id']) ? $question['cq_subject_id'] : null,
                    'qb_created_by' => $question['cq_created_by'] ?? $this->session->userdata('admin_id'),
                    'qb_created_on' => $question['cq_createdon'] ?? date('Y-m-d H:i:s'),
                    'qb_status' => $question['cq_status'],
                ];
                
                // Si hay opciones, convertirlas al nuevo formato
                if (isset($question['cq_options']) && !empty($question['cq_options'])) {
                    $new_question['qb_options'] = $question['cq_options'];
                }
                
                // Buscar si ya existe una pregunta igual en el banco (por texto y tipo)
                $this->db->where('qb_text', $question['cq_question']);
                $this->db->where('qb_type', $question['cq_type']);
                $existing = $this->db->get('question_bank')->row();
                if ($existing) {
                    $new_question_id = $existing->qb_id;
                } else {
                $new_question_id = $this->model_question_bank->save_question($new_question);
                }
                
                if ($new_question_id) {
                    // Guardar mapeo de ID antiguo a nuevo
                    $id_mapping[$question['cq_id']] = $new_question_id;
                    $stats['migrated_questions']++;
                    
                    // Verificar si la pregunta estaba asignada a un quiz
                    if (isset($question['cq_quiz_id']) && $question['cq_quiz_id'] > 0) {
                        $stats['assigned_questions']++;
                        
                        // Crear relación en la nueva tabla
                        $relation_data = [
                            'qq_quiz_id' => $question['cq_quiz_id'],
                            'qq_question_id' => $new_question_id,
                            'qq_order' => 0, // Orden por defecto
                            'qq_points' => $question['cq_points'] ?? 1,
                            'qq_created_on' => date('Y-m-d H:i:s')
                        ];
                        
                        $relation_id = $this->model_quiz_questions->assign_question(
                            $question['cq_quiz_id'],
                            $new_question_id,
                            $relation_data
                        );
                        
                        if ($relation_id) {
                            $stats['migrated_relations']++;
                        } else {
                            $stats['failed_relations']++;
                        }
                        
                        $stats['total_relations']++;
                    } else {
                        $stats['unassigned_questions']++;
                    }
                } else {
                    $stats['failed_questions']++;
                }
            } catch (Exception $e) {
                $stats['failed_questions']++;
                echo '<p style="color:red">Error migrando pregunta ID ' . $question['cq_id'] . ': ' . $e->getMessage() . '</p>';
            }
        }
        
        // Calcular tiempo total
        $end_time = microtime(true);
        $execution_time = ($end_time - $start_time);
        
        // Mostrar resultados
        echo '<h2>Resultados de la Migración</h2>';
        echo '<ul>';
        echo '<li><strong>Tiempo total:</strong> ' . number_format($execution_time, 2) . ' segundos</li>';
        echo '<li><strong>Preguntas totales:</strong> ' . $stats['total_questions'] . '</li>';
        echo '<li><strong>Preguntas migradas:</strong> ' . $stats['migrated_questions'] . '</li>';
        echo '<li><strong>Preguntas fallidas:</strong> ' . $stats['failed_questions'] . '</li>';
        echo '<li><strong>Preguntas sin asignar:</strong> ' . $stats['unassigned_questions'] . '</li>';
        echo '<li><strong>Preguntas asignadas a quizzes:</strong> ' . $stats['assigned_questions'] . '</li>';
        echo '<li><strong>Relaciones creadas:</strong> ' . $stats['migrated_relations'] . '</li>';
        echo '<li><strong>Relaciones fallidas:</strong> ' . $stats['failed_relations'] . '</li>';
        echo '</ul>';
        
        if ($stats['migrated_questions'] == $stats['total_questions']) {
            echo '<p style="color:green"><strong>Migración completada con éxito!</strong></p>';
        } else {
            echo '<p style="color:orange"><strong>Migración parcial - Revise los errores</strong></p>';
        }
        
        echo '<p><a href="' . site_url('admin/question_bank') . '" class="btn btn-primary">Ir al Banco de Preguntas</a></p>';
    }
    
    /**
     * Ejecutar la migraciu00f3n de esquema de base de datos para crear las tablas necesarias
     */
    public function migrate_schema() {
        $this->load->library('migration');
        
        if ($this->migration->current() === FALSE) {
            echo '<h1>Error en Migración</h1>';
            echo '<p>Error: ' . $this->migration->error_string() . '</p>';
        } else {
            echo '<h1>Migración Completada</h1>';
            echo '<p>Las tablas han sido creadas exitosamente.</p>';
            echo '<p><a href="' . site_url('admin/migrate_questions') . '" class="btn btn-primary">Continuar con la migración de datos</a></p>';
        }
    }
}
