<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

//Include Admin Wrapper. Break down things abit
include_once(APPPATH . "core/MY_Controller_Account.php");

class Quiz_result extends MY_Controller_Account {

    /**
     * Quiz Result Controller
     *
     * @package     Quiz Result
     * @version     1.0
     * @since       Version 1.0 2025
     */

    public $_list_data = array();
    private $course_id;
    private $subject_id;

    public function __construct() 
    {
        global $config;
        parent::__construct();

        $this->class_name = 'quiz_result';
        $this->view_pre = 'account/'.$this->class_name.'/';
        
        // Load penalties library
        $this->load->library('quiz_penalty_calculator');
        
        // Load required models
        $this->load->model('model_result');
        $this->load->model('model_quiz');
        $this->load->model('model_user');
        $this->load->model('model_course_quiz');
        
        $_POST = $this->input->post(NULL, false);
    }

    /**
     * Processes a quiz submission and applies penalties if applicable
     * 
     * @param int $quiz_id Quiz ID
     * @param int $user_id User ID
     * @param array $answers User's answers
     * @param float $score Score obtained
     * @param int $total_questions Total questions
     * @param int $correct_answers Correct answers
     * @param int $wrong_answers Wrong answers
     * @return array Processed result with penalties applied if applicable
     */
    public function process_quiz_submission($quiz_id, $user_id, $answers, $score, $total_questions, $correct_answers, $wrong_answers)
    {
        // Fecha actual para la entrega
        $submission_date = date('Y-m-d');
        
        // Calcular días de retraso
        $days_late = $this->quiz_penalty_calculator->calculate_days_late($quiz_id, $user_id, $submission_date);
        
        // Si hay retraso, calcular penalización
        $penalty_percentage = 0;
        $original_score = $score;
        $final_score = $score;
        $is_late = 0;
        
        if ($days_late > 0) {
            $is_late = 1;
            $penalty_percentage = $this->quiz_penalty_calculator->calculate_penalty_percentage($quiz_id, $user_id, $days_late);
            $final_score = $this->quiz_penalty_calculator->calculate_final_score($score, $penalty_percentage);
        }
        
        // Calcular puntos bonus (si hay preguntas con bonus)
        $bonus_points = $this->calculate_bonus_points($quiz_id, $answers);
        $final_score += $bonus_points;
        
        // Preparar datos para guardar
        $result_data = array();
        $result_data['r_quiz_id'] = $quiz_id;
        $result_data['r_user_id'] = $user_id;
        $result_data['r_total_point'] = $final_score;
        $result_data['r_obtain_point'] = $original_score;
        $result_data['r_correct_ans'] = $correct_answers;
        $result_data['r_wrong_ans'] = $wrong_answers;
        $result_data['r_total_question'] = $total_questions;
        $result_data['r_date'] = date('Y-m-d H:i:s');
        $result_data['r_status'] = 1;
        
        // Añadir campos de penalización
        $result_data['is_late_submission'] = $is_late;
        $result_data['days_late'] = $days_late;
        $result_data['penalty_applied'] = $penalty_percentage;
        $result_data['original_score'] = $original_score;
        $result_data['r_bonus_points'] = $bonus_points;
        
        // Calcular si el usuario aprobó o no
        $quiz_details = $this->model_quiz->find_by_pk($quiz_id);
        $passing_score = isset($quiz_details->quiz_passing_score) ? $quiz_details->quiz_passing_score : 60;
        $result_data['r_result'] = ($final_score >= $passing_score) ? 'pass' : 'fail';
        
        // Guardar resultado
        $this->model_result->set_attributes($result_data);
        $result_id = $this->model_result->save();
        
        // Preparar respuesta
        $response = array(
            'result_id' => $result_id,
            'original_score' => $original_score,
            'bonus_points' => $bonus_points,
            'is_late' => $is_late,
            'days_late' => $days_late,
            'penalty_percentage' => $penalty_percentage,
            'final_score' => $final_score,
            'result' => $result_data['r_result']
        );
        
        return $response;
    }
    
    /**
     * Calculates the bonus points earned in a quiz
     * 
     * @param int $quiz_id Quiz ID
     * @param array $answers User's answers
     * @return float Total bonus points earned
     */
    private function calculate_bonus_points($quiz_id, $answers)
    {
        $total_bonus = 0;
        
        // Get all quiz questions
        $params = array(
            'where' => array(
                'cq_quiz_id' => $quiz_id,
                'cq_is_bonus' => 1, // Only bonus questions
                'cq_status' => 1
            )
        );
        
        $bonus_questions = $this->model_course_quiz->find_all_active($params);
        
        if (!$bonus_questions) {
            return 0;
        }
        
        // For each bonus question, check if the answer is correct
        foreach ($bonus_questions as $question) {
            $question_id = $question->cq_id;
            
            // If there's no answer for this question, continue to the next one
            if (!isset($answers[$question_id])) {
                continue;
            }
            
            $user_answer = $answers[$question_id];
            $correct_answer = $question->cq_correct_answer;
            
            // Compare user's answer with correct answer
            if (is_array($user_answer)) {
                $user_answer = implode(',', $user_answer);
            }
            
            if ($user_answer == $correct_answer) {
                // Correct answer, add bonus points
                $total_bonus += floatval($question->cq_bonus_points);
            }
        }
        
        return $total_bonus;
    }
    
    /**
     * Muestra los resultados de un quiz específico
     * 
     * @param int $result_id ID del resultado
     */
    public function view($result_id)
    {
        $result = $this->model_result->find_by_pk($result_id);
        if (!$result) {
            redirect('account/dashboard', 'refresh');
        }
        $data = array();
        $data['result'] = $result;
        $data['quiz'] = $this->model_quiz->find_by_pk($result->r_quiz_id);
        $data['user'] = $this->model_user->find_by_pk($result->r_user_id);
        $data['has_penalty'] = $result->is_late_submission;
        $data['page_title'] = 'Quiz Result';
        // Fetch per-question answers using the new structure
        $this->load->model('model_result_quiz');
        $answers = $this->model_result_quiz->get_user_quiz_answers($result->r_user_id, $result->r_quiz_id);
        $data['results'] = $answers;
        // Fetch all questions for this quiz
        $this->load->model('model_quiz_questions');
        $questions_raw = $this->model_quiz_questions->get_quiz_questions($result->r_quiz_id);
        $questions = array();
        foreach ($questions_raw as $q) {
            $questions[$q['qb_id']] = $q;
        }
        $data['questions'] = $questions;
        $this->load_view('view', $data);
    }
    /**
     * Muestra el dashboard con todos los resultados del usuario actual
     */
    public function dashboard()
    {
        $user_id = $this->session->userdata('user_id');
        
        // Obtener todos los resultados del usuario
        $params = array(
            'where' => array(
                'r_user_id' => $user_id
            ),
            'order' => 'r_date DESC'
        );
        $results = $this->model_result->find_all_active($params);
        
        // Obtener información de todos los quices relacionados
        $quiz_ids = array();
        foreach ($results as $result) {
            $quiz_ids[] = $result->r_quiz_id;
        }
        
        $quizzes = array();
        if (!empty($quiz_ids)) {
            $quiz_params = array(
                'where_in' => array(
                    'quiz_id' => array_unique($quiz_ids)
                )
            );
            $quiz_data = $this->model_quiz->find_all_active($quiz_params);
            
            // Indexar por ID para fácil acceso
            foreach ($quiz_data as $quiz) {
                $quizzes[$quiz->quiz_id] = $quiz;
            }
        }
        
        // Preparar datos para la vista
        $data = array();
        $data['results'] = $results;
        $data['quizzes'] = $quizzes;
        $data['page_title'] = 'Dashboard de Resultados';
        
        $this->load_view('dashboard', $data);
    }
}

/* End of file Quiz_result.php */
/* Location: ./application/controllers/account/Quiz_result.php */
